/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.persistence;

import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Set;

/**
 * A bidirectional map that maintains both key-to-value and value-to-key mappings. This class provides an inverse view through the
 * {@link #inverse()} method.
 *
 * @param <K> the type of keys
 * @param <V> the type of values
 * @since 1.11
 */
final class BidirectionalMap<K, V> implements Map<K, V> {

  private final Map<K, V> forward;
  private final Map<V, K> backward;

  private BidirectionalMap(Map<K, V> forward, Map<V, K> backward) {
    this.forward = forward;
    this.backward = backward;
  }

  /**
   * Creates a new empty bidirectional map.
   *
   * @param <K> the type of keys
   * @param <V> the type of values
   * @return a new bidirectional map
   */
  public static <K, V> BidirectionalMap<K, V> create() {
    return new BidirectionalMap<>(new LinkedHashMap<>(), new LinkedHashMap<>());
  }

  @Override
  public int size() {
    return forward.size();
  }

  @Override
  public boolean isEmpty() {
    return forward.isEmpty();
  }

  @Override
  public boolean containsKey(Object key) {
    return forward.containsKey(key);
  }

  @Override
  public boolean containsValue(Object value) {
    return forward.containsValue(value);
  }

  @Override
  public V get(Object key) {
    return forward.get(key);
  }

  @Override
  public V put(K key, V value) {
    // Remove old mappings if they exist
    V oldValue = null;
    if (forward.containsKey(key)) {
      oldValue = forward.get(key);
      backward.remove(oldValue);
    }
    if (backward.containsKey(value)) {
      K oldKey = backward.get(value);
      forward.remove(oldKey);
    }

    forward.put(key, value);
    backward.put(value, key);

    return oldValue;
  }

  @Override
  public V remove(Object key) {
    V value = forward.remove(key);
    if (value != null) {
      backward.remove(value);
    }
    return value;
  }

  @Override
  public void putAll(Map<? extends K, ? extends V> m) {
    for (Entry<? extends K, ? extends V> entry : m.entrySet()) {
      put(entry.getKey(), entry.getValue());
    }
  }

  @Override
  public void clear() {
    forward.clear();
    backward.clear();
  }

  @Override
  public Set<K> keySet() {
    return forward.keySet();
  }

  @Override
  public Collection<V> values() {
    return forward.values();
  }

  @Override
  public Set<Entry<K, V>> entrySet() {
    return forward.entrySet();
  }

  /**
   * Returns the inverse view of this bidirectional map. Changes to the inverse map are reflected in this map.
   *
   * @return the inverse bidirectional map
   */
  public BidirectionalMap<V, K> inverse() {
    return new BidirectionalMap<>(backward, forward);
  }
}

