/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.persistence.reduced;

import static java.util.Optional.empty;
import static org.mule.metadata.persistence.MetadataTypeConstants.ANNOTATIONS;
import static org.mule.metadata.persistence.MetadataTypeConstants.TYPE_ID;

import org.mule.metadata.api.builder.BaseTypeBuilder;
import org.mule.metadata.api.builder.TypeBuilder;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.api.model.ObjectType;
import org.mule.metadata.persistence.JsonMetadataTypeLoader;
import org.mule.metadata.persistence.ObjectTypeReferenceHandler;

import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.Stack;

import com.google.gson.JsonElement;
import com.google.gson.JsonObject;

/**
 * {@link JsonMetadataTypeLoader} extension that knows how to load types from a catalog
 * considering recursive cases between different objects in the catalog.
 *
 * e.g.: A -> B and B -> A both described in the catalog.
 *
 * @since 1.2.0, 1.1.7
 */
class CatalogJsonMetadataTypeLoader extends JsonMetadataTypeLoader {

  /**
   * Using a map as stack, only comparing by the key.
   */
  private Map<String, TypeBuilder> typeIdBuilderStack;

  CatalogJsonMetadataTypeLoader(ObjectTypeReferenceHandler referenceHandler) {
    super(referenceHandler);
  }

  /**
   * Parses an {@link ObjectType} catalog entry in a JSON representation and returns a new {@link MetadataType} instance.
   *
   * @param jsonElement {@link ObjectType} JSON representation from the catalog as a {@link JsonElement}
   * @return {@link Optional} value of a {@link MetadataType}
   */
  @Override
  public Optional<MetadataType> load(JsonElement jsonElement) {
    return load(jsonElement, new HashMap<>()).map(TypeBuilder::build);
  }

  public Optional<TypeBuilder> load(JsonElement jsonElement, Map<String, TypeBuilder> stack) {
    typeIdBuilderStack = stack;
    return super.load(jsonElement, new Stack<>(), new Stack<>());
  }

  public TypeBuilder buildType(JsonElement metadataType, BaseTypeBuilder baseBuilder) {
    Optional<String> foundId = findId(metadataType);
    foundId.ifPresent(id -> typeIdBuilderStack.put(id, baseBuilder));
    TypeBuilder typeBuilder = super.buildType(metadataType, baseBuilder);
    foundId.ifPresent(id -> typeIdBuilderStack.remove(id));
    return typeBuilder;
  }

  private Optional<String> findId(JsonElement element) {
    if (element.isJsonObject() && element.getAsJsonObject().has(ANNOTATIONS)) {
      JsonObject annotations = element.getAsJsonObject().get(ANNOTATIONS).getAsJsonObject();
      if (annotations.has(TYPE_ID)) {
        JsonElement typeId = annotations.get(TYPE_ID);
        return Optional.of(typeId.getAsString());
      }
    }
    return empty();
  }
}
