/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.persistence.reduced;

import static java.util.Optional.empty;
import static org.mule.metadata.api.utils.MetadataTypeUtils.getTypeId;
import static org.mule.metadata.persistence.MetadataTypeConstants.REF_FLAG;
import static org.mule.metadata.persistence.MetadataTypeConstants.TYPE;

import org.mule.metadata.api.builder.TypeBuilder;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.api.model.ObjectType;
import org.mule.metadata.persistence.MetadataSerializingException;
import org.mule.metadata.persistence.ObjectTypeReferenceHandler;

import java.io.IOException;
import java.util.Collections;
import java.util.Map;
import java.util.Optional;

import com.google.gson.stream.JsonWriter;

/**
 * {@link ObjectTypeReferenceHandler} implementation for type seralization
 * that declare an inner type catalog in its serialized structure.
 *
 * @since 1.2.0, 1.1.7
 */
public class InnerCatalogObjectTypeReferenceHandler implements ObjectTypeReferenceHandler {

  private final Map<String, MetadataType> catalog;

  InnerCatalogObjectTypeReferenceHandler() {
    this.catalog = Collections.emptyMap();
  }

  InnerCatalogObjectTypeReferenceHandler(Map<String, MetadataType> catalog) {
    this.catalog = catalog;
  }

  /**
   * If a ref is found, the type builder is taken from the previously populated catalog.
   */
  @Override
  public Optional<TypeBuilder> readReference(String typeReference) {
    if (typeReference.startsWith(REF_FLAG)) {
      return Optional.of(() -> catalog.get(typeReference.replace(REF_FLAG, "")));
    }
    return empty();
  }

  /**
   * Only should write references of types than have a typeId to be referenced to.
   */
  @Override
  public boolean shouldWriteReference(ObjectType type) {
    return getTypeId(type).isPresent();
  }

  /**
   * Writes a type reference if this has an associated type id to be later fetched from the catalog.
   */
  @Override
  public Optional<String> writeReference(ObjectType type, JsonWriter writer) {
    Optional<String> typeId = getTypeId(type);
    if (typeId.isPresent()) {
      try {
        writer.name(TYPE).value(REF_FLAG + typeId.get());
      } catch (IOException e) {
        throw new MetadataSerializingException("MetadataType reference [" + typeId.get() + "]", e);
      }
    }
    return typeId;
  }

}
