/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.persistence.reduced;

import static java.util.Collections.emptyMap;
import static org.mule.metadata.persistence.MetadataTypeConstants.CATALOG;
import static org.mule.metadata.persistence.MetadataTypeConstants.VALUE;

import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.persistence.JsonMetadataTypeLoader;
import org.mule.metadata.persistence.SerializedMetadataTypeLoader;

import java.util.Collections;
import java.util.Map;
import java.util.Optional;

import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;

/**
 * {@link SerializedMetadataTypeLoader} that creates instances of {@link MetadataType} from JSON representations generated by
 * {@link ReducedJsonMetadataTypeWriter}.
 * <p>
 * Instances of this class are not thread safe, so only one deserialization at a time can be made with each instance.
 *
 * @since 1.2.0, 1.1.7
 */
public class ReducedJsonMetadataTypeLoader implements SerializedMetadataTypeLoader {

  @Override
  public Optional<MetadataType> load(String serialized) {
    final JsonElement element = new JsonParser().parse(serialized);
    return loadFromElement(element);
  }

  @Override
  public Optional<MetadataType> load(JsonElement jsonElement) {
    return loadFromElement(jsonElement);
  }

  private Optional<MetadataType> loadFromElement(JsonElement element) {
    JsonObject value = element.getAsJsonObject().get(VALUE).getAsJsonObject();
    JsonElement catalogElement = element.getAsJsonObject().get(CATALOG);
    return new JsonMetadataTypeLoader(new InnerCatalogObjectTypeReferenceHandler(getCatalog(catalogElement))).load(value);
  }

  private Map<String, MetadataType> getCatalog(JsonElement catalogElement) {
    return catalogElement != null ? MetadataTypeCatalogLoader.loadCatalog(catalogElement.getAsJsonObject()) : emptyMap();
  }
}
