/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.persistence.reduced;


import static java.util.Optional.empty;
import static java.util.Optional.of;
import static java.util.Optional.ofNullable;
import static java.util.stream.Collectors.toMap;
import static org.mule.metadata.persistence.MetadataTypeConstants.REF_FLAG;

import org.mule.metadata.api.builder.TypeBuilder;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.api.model.ObjectType;
import org.mule.metadata.persistence.ObjectTypeReferenceHandler;

import java.util.HashMap;
import java.util.Map;
import java.util.Optional;

import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.stream.JsonWriter;

/**
 * Utility class that loads an {@link ObjectType}s catalog (as a Map) from a
 * serialized representation {@link JsonObject}.
 *
 * @since 1.2.0, 1.1.7
 */
class MetadataTypeCatalogLoader {

  static Map<String, MetadataType> loadCatalog(JsonObject jsonCatalog) {
    Map<String, TypeBuilder> catalog = new HashMap<>();
    Map<String, TypeBuilder> stack = new HashMap<>();
    jsonCatalog.entrySet().forEach(e -> loadEntry(catalog, stack, jsonCatalog, e.getKey()));
    return catalog.entrySet().stream().collect(toMap(Map.Entry::getKey, e -> e.getValue().build()));
  }

  private static TypeBuilder loadEntry(Map<String, TypeBuilder> catalog,
                                       Map<String, TypeBuilder> stack,
                                       JsonObject jsonCatalog, String key) {

    if (catalog.containsKey(key)) {
      return catalog.get(key);
    }

    CatalogJsonMetadataTypeLoader loader = new CatalogJsonMetadataTypeLoader(new ObjectTypeReferenceHandler() {

      @Override
      public Optional<String> writeReference(ObjectType type, JsonWriter writer) {
        // do nothing
        return empty();
      }

      @Override
      public Optional<TypeBuilder> readReference(String typeRef) {

        if (typeRef.startsWith(REF_FLAG)) {
          String id = typeRef.replace(REF_FLAG, "");

          if (stack.containsKey(id)) {
            return ofNullable(stack.get(id));
          }

          TypeBuilder metadataType = loadEntry(catalog, stack, jsonCatalog, id);
          return of(metadataType);
        }
        return empty();
      }
    }, jsonCatalog);
    JsonElement catalogTypeElement = jsonCatalog.get(key);
    if (catalogTypeElement != null) {
      Optional<TypeBuilder> loadedBuilder = loader.load(catalogTypeElement, stack);
      if (loadedBuilder.isPresent()) {
        TypeBuilder typeBuilder = loadedBuilder.get();
        catalog.put(key, typeBuilder);
        return typeBuilder;
      }
    }
    throw new RuntimeException("Element [" + key + "] not found in catalog");
  }
}
