/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.persistence.reduced;

import static org.mule.metadata.persistence.MetadataTypeConstants.CATALOG;
import static org.mule.metadata.persistence.MetadataTypeConstants.OBJECT;
import static org.mule.metadata.persistence.MetadataTypeConstants.VALUE;

import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.api.model.ObjectType;
import org.mule.metadata.api.visitor.MetadataTypeVisitor;
import org.mule.metadata.persistence.AbstractJsonMetadataTypeWriter;
import org.mule.metadata.persistence.MetadataSerializingException;
import org.mule.metadata.persistence.MetadataTypeWriter;
import org.mule.metadata.persistence.ObjectTypeReferenceHandler;
import org.mule.metadata.persistence.serializer.ObjectTypeSerializer;
import org.mule.metadata.persistence.serializer.TypeSerializerVisitor;

import java.io.IOException;
import java.util.Map;
import java.util.Stack;

import com.google.gson.stream.JsonWriter;

/**
 * {@link MetadataTypeWriter} implementation that serializes a {@link MetadataType} object into a JSON
 * object with a reduced structure by creating a type catalog for all {@link ObjectType}s found.
 * <p>
 * The end serialization ends up with an structure like the one described in the following snippet.
 * <p>
 * /**
 * <pre>
 * {
 *   "value": {...}
 *   "catalog": {...}
 * }
 * </pre>
 * <p>
 * The value element contains the serializing {@link MetadataType} structure with references to the catalog. NOTE that if
 * the serializing object is an object type, then a simple reference to the catalog will be described in the value field.
 * e.g.: @{code ... "value"{ type: "@ref:myObjectTypeId" } ...}
 * <p>
 * The catalog contains all the described object types in the root {@link MetadataType}, also, {@link ObjectType}s described
 * in the catalog can also contain references to other types declared in that catalog.
 *
 * @since 1.2.0, 1.1.7
 */
public class ReducedJsonMetadataTypeWriter extends AbstractJsonMetadataTypeWriter {

  /**
   * {@inheritDoc}
   */
  @Override
  protected void write(MetadataType type) throws IOException {
    CatalogTypeCollectorMetadataTypeVisitor collector = new CatalogTypeCollectorMetadataTypeVisitor();
    ObjectTypeReferenceHandler referenceHandler = new InnerCatalogObjectTypeReferenceHandler(collector);
    type.accept(collector);
    writer.beginObject().name(VALUE);
    writeModel(type, referenceHandler);
    writeCatalog(writer, collector.getCatalog(), referenceHandler);
    writer.endObject();
  }

  private void writeModel(MetadataType type, ObjectTypeReferenceHandler referenceHandler) {
    type.accept(new MetadataTypeVisitor() {

      @Override
      protected void defaultVisit(MetadataType metadataType) {
        type.accept(new TypeSerializerVisitor(writer, referenceHandler, typeStack, false));
      }
    });
  }

  private void writeCatalog(JsonWriter writer, Map<String, ObjectType> registeredTypes,
                            ObjectTypeReferenceHandler referenceHandler)
      throws IOException {
    if (!registeredTypes.isEmpty()) {
      writer.name(CATALOG);
      writer.beginObject();
      registeredTypes.forEach((id, type) -> {
        try {
          writer.name(id);
          TypeSerializerVisitor visitor = new TypeSerializerVisitor(writer, referenceHandler, new Stack<>(), false);
          new ObjectTypeSerializer(visitor).serialize(writer, type, typeStack);
        } catch (IOException e) {
          throw new MetadataSerializingException(OBJECT + " MetadataType for catalog");
        }
      });
      writer.endObject();
    }
  }

  @Override
  public ReducedJsonMetadataTypeWriter setPrettyPrint(boolean prettyPrint) {
    super.setPrettyPrint(prettyPrint);
    return this;
  }
}
