/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.persistence.serializer;

import static java.util.stream.Collectors.toList;
import static org.mule.metadata.persistence.MetadataTypeConstants.ANNOTATIONS;
import static org.mule.metadata.persistence.MetadataTypeConstants.FORMAT;
import static org.mule.metadata.persistence.MetadataTypeConstants.ID;
import static org.mule.metadata.persistence.MetadataTypeConstants.LABEL;
import static org.mule.metadata.persistence.MetadataTypeConstants.TYPE;
import static org.mule.metadata.persistence.MetadataTypeConstants.VALID_MIME_TYPES;
import static org.mule.metadata.persistence.MetadataTypeConstants.commonMetadataFormats;

import org.mule.metadata.api.annotation.TypeAnnotation;
import org.mule.metadata.api.model.MetadataFormat;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.persistence.MetadataSerializingException;
import org.mule.metadata.persistence.TypeAnnotationSerializer;
import org.mule.metadata.persistence.TypeAnnotationSerializerFactory;

import java.io.IOException;
import java.util.Collection;
import java.util.List;
import java.util.Stack;

import com.google.gson.stream.JsonWriter;

/**
 * Base implementation for {@link TypeSerializer}s, knows how to write all the common parts of
 * {@link MetadataType}s such as annotations, format, etc.
 *
 * @param <T> the specific {@link MetadataType}
 * @since 1.2.0, 1.1.7
 */
public abstract class BaseTypeSerializer<T extends MetadataType> implements TypeSerializer<T> {

  protected final String typeName;

  private final TypeAnnotationSerializer typeAnnotationSerializer;

  BaseTypeSerializer(String typeName) {
    this.typeName = typeName;
    this.typeAnnotationSerializer = TypeAnnotationSerializerFactory.getInstance().getTypeAnnotationSerializer();
  }

  @Override
  public void serialize(JsonWriter writer, T type, Stack<MetadataType> typeStack) {
    try {
      writer.beginObject();
      writeMetadataFormat(writer, type, typeStack);
      typeStack.push(type);
      writeType(writer, type, typeName);
      doSerialize(writer, type, typeStack);
      writer.endObject();
      typeStack.pop();
    } catch (IOException e) {
      throw new MetadataSerializingException(typeName + " MetadataType", e);
    }
  }

  protected abstract void doSerialize(JsonWriter writer, T type, Stack<MetadataType> typeStack) throws IOException;

  protected void writeMetadataFormat(JsonWriter writer, MetadataType type, Stack<MetadataType> typeStack) throws IOException {
    final MetadataFormat metadataFormat = type.getMetadataFormat();
    if (typeStack.isEmpty() || metadataFormat != typeStack.peek().getMetadataFormat()) {
      writer.name(FORMAT);
      if (commonMetadataFormats.contains(metadataFormat)) {
        writer.value(metadataFormat.getId());
      } else {
        writer.beginObject();
        writer.name(ID).value(metadataFormat.getId());
        if (metadataFormat.getLabel().isPresent()) {
          writer.name(LABEL).value(metadataFormat.getLabel().get());
        }
        writer.name(VALID_MIME_TYPES);
        writer.beginArray();
        for (String s : metadataFormat.getValidMimeTypes()) {
          writer.value(s);
        }
        writer.endArray();
        writer.endObject();
      }
    }
  }

  private void writeType(JsonWriter writer, MetadataType metadataType, String type) {
    try {
      writer.name(TYPE).value(type);
      writeAnnotations(writer, metadataType.getAnnotations());
    } catch (IOException e) {
      throw new MetadataSerializingException(type, e);
    }
  }

  private String getAnnotationJsonName(TypeAnnotation annotation) {
    if (typeAnnotationSerializer.getNameClassMapping().containsKey(annotation.getName())) {
      return annotation.getName();
    }
    return annotation.getClass().getName();
  }

  protected void writeAnnotations(JsonWriter writer, Collection<TypeAnnotation> annotations) throws IOException {
    List<TypeAnnotation> publicAnnotations = annotations
        .stream()
        .filter(TypeAnnotation::isPublic)
        .collect(toList());

    if (!publicAnnotations.isEmpty()) {
      writer.name(ANNOTATIONS);
      writer.beginObject();

      for (TypeAnnotation annotation : publicAnnotations) {
        writer.name(getAnnotationJsonName(annotation));
        typeAnnotationSerializer.serialize(writer, annotation);
      }
      writer.endObject();
    }
  }
}
