/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.persistence.serializer;

import static org.mule.metadata.persistence.MetadataTypeConstants.FUNCTION;
import static org.mule.metadata.persistence.MetadataTypeConstants.NAME;
import static org.mule.metadata.persistence.MetadataTypeConstants.PARAMETERS;
import static org.mule.metadata.persistence.MetadataTypeConstants.REQUIRED;
import static org.mule.metadata.persistence.MetadataTypeConstants.RETURN_TYPE;
import static org.mule.metadata.persistence.MetadataTypeConstants.TYPE;

import org.mule.metadata.api.model.FunctionParameter;
import org.mule.metadata.api.model.FunctionType;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.persistence.MetadataSerializingException;

import java.io.IOException;
import java.util.Collection;
import java.util.Stack;

import com.google.gson.stream.JsonWriter;

/**
 * {@link TypeSerializer} for {@link FunctionType}s.
 *
 * @since 1.2.0, 1.1.7
 */
public class FunctionTypeSerializer extends AbstractComplexTypeSerializer<FunctionType> {

  private final TypeSerializer<MetadataType> delegate;

  FunctionTypeSerializer(TypeSerializer<MetadataType> delegate) {
    super(FUNCTION);
    this.delegate = delegate;
  }

  @Override
  public void doSerialize(JsonWriter writer, FunctionType functionType, Stack<MetadataType> typeStack) throws IOException {
    final Collection<FunctionParameter> parameters = functionType.getParameters();
    writer.name(PARAMETERS);
    writer.beginArray();

    for (FunctionParameter parameter : parameters) {
      writer.beginObject();
      writer.name(NAME).value(parameter.getName());
      if (parameter.isOptional()) {
        writer.name(REQUIRED).value(!parameter.isOptional());
      }
      writer.name(TYPE);
      delegate.serialize(writer, parameter.getType(), typeStack);
      writer.endObject();
    }
    writer.endArray();

    functionType.getReturnType().ifPresent(returnType -> {
      try {
        writer.name(RETURN_TYPE);
        delegate.serialize(writer, returnType, typeStack);
      } catch (IOException e) {
        throw new MetadataSerializingException("the returnType field for FunctionType", e);
      }
    });
  }
}
