/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.persistence.serializer;

import static org.mule.metadata.persistence.MetadataTypeConstants.ATTRIBUTES;
import static org.mule.metadata.persistence.MetadataTypeConstants.FIELDS;
import static org.mule.metadata.persistence.MetadataTypeConstants.KEY;
import static org.mule.metadata.persistence.MetadataTypeConstants.MODEL;
import static org.mule.metadata.persistence.MetadataTypeConstants.NAME;
import static org.mule.metadata.persistence.MetadataTypeConstants.OBJECT;
import static org.mule.metadata.persistence.MetadataTypeConstants.OPEN;
import static org.mule.metadata.persistence.MetadataTypeConstants.ORDERED;
import static org.mule.metadata.persistence.MetadataTypeConstants.REPEATED;
import static org.mule.metadata.persistence.MetadataTypeConstants.REQUIRED;
import static org.mule.metadata.persistence.MetadataTypeConstants.TRUE;

import org.mule.metadata.api.annotation.RegexPatternAnnotation;
import org.mule.metadata.api.annotation.TypeAnnotation;
import org.mule.metadata.api.model.AttributeFieldType;
import org.mule.metadata.api.model.AttributeKeyType;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.api.model.ObjectFieldType;
import org.mule.metadata.api.model.ObjectKeyType;
import org.mule.metadata.api.model.ObjectType;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.Stack;

import com.google.gson.stream.JsonWriter;

/**
 * {@link TypeSerializer} for {@link ObjectType}s
 *
 * @since 1.2.0, 1.1.7
 */
public class ObjectTypeSerializer extends AbstractComplexTypeSerializer<ObjectType> {

  private final TypeSerializer<MetadataType> delegate;

  public ObjectTypeSerializer(TypeSerializer<MetadataType> delegate) {
    super(OBJECT);
    this.delegate = delegate;
  }

  @Override
  public void doSerialize(JsonWriter writer, ObjectType objectType, Stack<MetadataType> typeStack) throws IOException {
    if (objectType.isOrdered()) {
      writer.name(ORDERED).value(true);
    }
    if (objectType.isOpen()) {
      writer.name(OPEN);
      delegate.serialize(writer, objectType.getOpenRestriction().get(), typeStack);
    }
    writer.name(FIELDS);
    writer.beginArray();
    for (ObjectFieldType field : objectType.getFields()) {
      writeObjectField(writer, typeStack, field);
    }
    writer.endArray();
  }

  private void writeObjectField(JsonWriter writer, Stack<MetadataType> typeStack, ObjectFieldType field) throws IOException {
    writer.beginObject();
    final ObjectKeyType key = field.getKey();
    final Collection<TypeAnnotation> keyAnnotations = key.getAnnotations();

    final HashMap<String, String> stringObjectHashMap = new HashMap<>();
    if (field.isRequired()) {
      stringObjectHashMap.put(REQUIRED, TRUE);
    }
    if (field.isRepeated()) {
      stringObjectHashMap.put(REPEATED, TRUE);
    }
    createKeyObject(writer, keyAnnotations, field.getKey(), stringObjectHashMap, typeStack);
    writer.name(MODEL);
    delegate.serialize(writer, field.getValue(), typeStack);
    writeAnnotations(writer, field.getAnnotations());
    writer.endObject();
  }

  private void createKeyObject(JsonWriter writer, Collection<TypeAnnotation> keyAnnotations,
                               ObjectKeyType key, HashMap<String, String> additionalProperties,
                               Stack<MetadataType> stack)
      throws IOException {
    writer.name(KEY);
    writer.beginObject();
    writer.name(NAME);
    String keyString;
    if (key.isName()) {
      keyString = key.getName().toString();
    } else {
      keyString = key.getPattern().toString();
      final RegexPatternAnnotation patternAnnotation = new RegexPatternAnnotation(keyString);
      if (!keyAnnotations.contains(patternAnnotation)) {
        keyAnnotations.add(patternAnnotation);
      }
    }
    writer.value(keyString);
    writeAnnotations(writer, keyAnnotations);
    createAttributes(writer, key, stack);
    for (Map.Entry<String, String> property : additionalProperties.entrySet()) {
      writer.name(property.getKey()).value(property.getValue());
    }

    writer.endObject();
  }

  private void createAttributes(JsonWriter writer, ObjectKeyType key, Stack<MetadataType> stack) throws IOException {
    final Collection<AttributeFieldType> attributes = key.getAttributes();
    if (!attributes.isEmpty()) {
      writer.name(ATTRIBUTES);
      writer.beginArray();
      for (AttributeFieldType attribute : attributes) {
        createAttribute(writer, attribute, stack);
      }
      writer.endArray();
    }
  }

  private Map<String, String> getAttributeAdditionalProperties(AttributeFieldType attribute) {
    final HashMap<String, String> result = new HashMap<>();
    if (attribute.isRequired()) {
      result.put(REQUIRED, TRUE);
    }
    return result;
  }

  private void createAttribute(JsonWriter writer, AttributeFieldType attribute, Stack<MetadataType> stack) throws IOException {
    writer.beginObject();
    final AttributeKeyType attributeKey = attribute.getKey();
    createAttributeKeyObject(writer, attributeKey, getAttributeAdditionalProperties(attribute));
    writer.name(MODEL);
    delegate.serialize(writer, attribute.getValue(), stack);
    writeAnnotations(writer, attribute.getAnnotations());
    writer.endObject();
  }

  private void createAttributeKeyObject(JsonWriter writer, AttributeKeyType key,
                                        Map<String, String> additionalProperties)
      throws IOException {
    writer.name(KEY);
    writer.beginObject();
    writer.name(NAME);
    String keyString;
    Collection<TypeAnnotation> keyAnnotations = new ArrayList<>();
    if (key.isName()) {
      keyString = key.getName().toString();
    } else {
      keyString = key.getPattern().toString();
      final RegexPatternAnnotation patternAnnotation = new RegexPatternAnnotation(keyString);
      if (!keyAnnotations.contains(patternAnnotation)) {
        keyAnnotations.add(patternAnnotation);
      }
    }
    writer.value(keyString);
    writeAnnotations(writer, keyAnnotations);
    for (Map.Entry<String, String> property : additionalProperties.entrySet()) {
      writer.name(property.getKey()).value(property.getValue());
    }
    writer.endObject();
  }
}
