/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.persistence.reduced;

import org.mule.metadata.api.builder.BaseTypeBuilder;
import org.mule.metadata.api.builder.TypeBuilder;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.api.model.ObjectType;
import org.mule.metadata.persistence.JsonMetadataTypeLoader;
import org.mule.metadata.persistence.ObjectTypeReferenceHandler;

import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.Stack;

import com.google.gson.JsonElement;
import com.google.gson.JsonObject;

/**
 * {@link JsonMetadataTypeLoader} extension that knows how to load types from a catalog
 * considering recursive cases between different objects in the catalog.
 *
 * e.g.: A -> B and B -> A both described in the catalog.
 *
 * @since 1.2.0, 1.1.7
 */
class CatalogJsonMetadataTypeLoader extends JsonMetadataTypeLoader {

  /**
   * Uses the {@link JsonObject} raw catalog to identify each entry when recursive elements are found.
   */
  private JsonObject jsonCatalog;
  /**
   * Using a map as stack, only comparing by the key.
   */
  private Map<String, TypeBuilder> typeIdBuilderStack;

  CatalogJsonMetadataTypeLoader(ObjectTypeReferenceHandler referenceHandler, JsonObject jsonCatalog) {
    super(referenceHandler);
    this.jsonCatalog = jsonCatalog;
  }

  /**
   * Parses an {@link ObjectType} catalog entry in a JSON representation and returns a new {@link MetadataType} instance.
   *
   * @param jsonElement {@link ObjectType} JSON representation from the catalog as a {@link JsonElement}
   * @return {@link Optional} value of a {@link MetadataType}
   */
  @Override
  public Optional<MetadataType> load(JsonElement jsonElement) {
    return load(jsonElement, new HashMap<>()).map(TypeBuilder::build);
  }

  public Optional<TypeBuilder> load(JsonElement jsonElement, Map<String, TypeBuilder> stack) {
    typeIdBuilderStack = stack;
    return super.load(jsonElement, new Stack<>(), new Stack<>());
  }

  public TypeBuilder buildType(JsonElement metadataType, BaseTypeBuilder baseBuilder) {
    Optional<String> foundId = findId(metadataType);
    foundId.ifPresent(id -> typeIdBuilderStack.put(id, baseBuilder));
    TypeBuilder typeBuilder = super.buildType(metadataType, baseBuilder);
    foundId.ifPresent(id -> typeIdBuilderStack.remove(id));
    return typeBuilder;
  }

  private Optional<String> findId(JsonElement element) {
    return jsonCatalog.entrySet().stream().filter(entry -> entry.getValue().equals(element)).map(entry -> entry.getKey())
        .findFirst();
  }
}
