/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.persistence.type.adapter;

import org.mule.metadata.api.annotation.EnumAnnotation;
import org.mule.metadata.api.annotation.TypeAnnotation;

import java.io.IOException;

import com.google.gson.Gson;
import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import com.google.gson.TypeAdapter;
import com.google.gson.stream.JsonReader;
import com.google.gson.stream.JsonWriter;

/**
 * {@link TypeAdapter} implementation to be able to serialize {@link EnumAnnotation} instances. This {@link TypeAnnotation}
 * requires a special {@link TypeAdapter} due that Enum can be of an generic type.
 * <p>
 * To not loose the Type information, this {@link TypeAdapter} serializes the generic type of the Enum to be able to create
 * the same instance at deserialization time.
 *
 * @since 1.0
 */
public final class EnumTypeAnnotationTypeAdapter extends TypeAdapter<EnumAnnotation> {

  private static final String TYPE = "type";
  private static final String VALUES = "values";
  private static final Gson GSON = new Gson();

  @Override
  public void write(JsonWriter out, EnumAnnotation value) throws IOException {
    final Class<?> componentType = value.getValues().getClass().getComponentType();
    final Class<?> arrayType = value.getValues().getClass();
    out.beginObject();
    out.name(TYPE);
    out.value(arrayType.getName());
    out.name(VALUES);
    out.beginArray();
    for (Object object : value.getValues()) {
      GSON.toJson(object, componentType, out);
    }
    out.endArray();
    out.endObject();
  }

  @Override
  public EnumAnnotation read(JsonReader in) throws IOException {
    final JsonElement parse = new JsonParser().parse(in);
    final JsonObject enumAnnotation = parse.getAsJsonObject();


    final String arrayType = enumAnnotation.get(TYPE).getAsString();
    final JsonArray iterator = enumAnnotation.get(VALUES).getAsJsonArray();
    Object array;
    try {
      final Class<?> aClass = Class.forName(arrayType);
      array = GSON.fromJson(iterator, aClass);
    } catch (ClassNotFoundException e) {
      array = GSON.fromJson(iterator, Object[].class);
    }
    return new EnumAnnotation((Object[]) array);
  }
}
