/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.persistence.serializer;

import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.api.model.ObjectType;
import org.mule.metadata.api.model.TupleType;
import org.mule.metadata.persistence.MetadataSerializingException;
import org.mule.metadata.persistence.ObjectTypeReferenceHandler;

import java.io.IOException;
import java.util.Stack;

import com.google.gson.stream.JsonWriter;

/**
 * {@link TypeSerializer} for {@link TupleType}s.
 *
 * @since 1.2.0, 1.1.7
 */
public class TypeReferenceSerializer extends BaseTypeSerializer<ObjectType> {

  private final ObjectTypeReferenceHandler handler;

  /**
   * This is added to maintain backwards compatibility, when writing ObjectType references
   * the format sometimes is required to appear in the generated reference element. This is redundant
   * since the reference definition also contains the format but we need to maintain the old structure.
   */
  private final boolean formatOnReferences;

  TypeReferenceSerializer(ObjectTypeReferenceHandler handler, boolean formatOnReferences) {
    super("");
    this.handler = handler;
    this.formatOnReferences = formatOnReferences;
  }

  @Override
  public void serialize(JsonWriter writer, ObjectType type, Stack<MetadataType> typeStack) {
    try {
      writer.beginObject();
      if (formatOnReferences) {
        writeMetadataFormat(writer, type, typeStack);
      }
      doSerialize(writer, type, typeStack);
      writer.endObject();
    } catch (IOException e) {
      throw new MetadataSerializingException("ObjectType reference", e);
    }
  }

  @Override
  protected void doSerialize(JsonWriter writer, ObjectType type, Stack<MetadataType> typeStack) {
    handler.writeReference(type, writer);
  }
}
