/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.persistence;


import org.mule.metadata.api.model.MetadataType;

import java.io.IOException;
import java.util.Optional;

import com.google.gson.Gson;
import com.google.gson.JsonElement;
import com.google.gson.JsonParser;
import com.google.gson.TypeAdapter;
import com.google.gson.stream.JsonReader;
import com.google.gson.stream.JsonWriter;

/**
 * Base {@link Gson} {@link TypeAdapter<MetadataType>} that helps to serialize and deserialize {@link MetadataType} objects.
 * <p>
 * {@link TypeAdapter} implementation is done in this base class, concrete implementations should provide a
 * {@link MetadataTypeWriter} and a {@link SerializedMetadataTypeLoader} implementations that actually know how to
 * serialize/deserialize the {@link MetadataType} in some JSON structure.
 *
 * @since 1.2.0, 1.1.7
 */
public abstract class BaseMetadataTypeGsonTypeAdapter extends TypeAdapter<MetadataType> {

  private final MetadataTypeWriter jsonMetadataTypeWriter;
  private final SerializedMetadataTypeLoader jsonMetadataTypeLoader;
  private boolean allowNullInput = false;

  public BaseMetadataTypeGsonTypeAdapter(MetadataTypeWriter writer, SerializedMetadataTypeLoader loader) {
    this.jsonMetadataTypeWriter = writer;
    this.jsonMetadataTypeLoader = loader;
  }

  /**
   * Given a {@link MetadataType} as parameter, this serializes it, and writes the Json object using the {@param jsonWriter}
   *
   * @param jsonWriter   {@link JsonWriter} instance to write the serialized object into Json
   * @param metadataType {@link MetadataType} instance to doSerialize
   * @throws IOException
   */
  @Override
  public void write(JsonWriter jsonWriter, MetadataType metadataType) throws IOException {
    jsonMetadataTypeWriter.write(metadataType, jsonWriter);
  }

  /**
   * Uses the {@link JsonReader} containing the JSON representation of a {@link MetadataType}, it gets deserialized and returns
   * the instance.
   *
   * @param jsonReader which contains as root level the JSON representation of a {@link MetadataType}
   * @return a {@link MetadataType} instance that represents the deserialized value of the JSON object.
   * @throws IOException
   */
  @Override
  public MetadataType read(JsonReader jsonReader) throws IOException {
    final JsonElement jsonMetadataTypeElement = new JsonParser().parse(jsonReader);

    if (jsonMetadataTypeElement.isJsonNull() && allowNullInput) {
      return null;
    }

    final Optional<MetadataType> load = jsonMetadataTypeLoader.load(jsonMetadataTypeElement);

    if (load.isPresent()) {
      return load.get();
    }

    // TODO - review, this case won't never happen
    throw new MetadataDeserializingException("Unknown error happen deserializing the MetadataType object");
  }

  public void setAllowNullInput(boolean allowNullInput) {
    this.allowNullInput = allowNullInput;
  }
}
