/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.persistence.reduced;

import static java.util.Optional.empty;
import static java.util.Optional.of;
import static org.mule.metadata.api.utils.MetadataTypeUtils.getTypeId;

import org.mule.metadata.api.model.ArrayType;
import org.mule.metadata.api.model.FunctionType;
import org.mule.metadata.api.model.IntersectionType;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.api.model.ObjectType;
import org.mule.metadata.api.model.UnionType;
import org.mule.metadata.api.visitor.MetadataTypeVisitor;

import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

/**
 * {@link MetadataTypeVisitor} implementation that registers all the {@link ObjectType}s declared inside another
 * {@link MetadataType} if any to create a type catalog.
 * <p>
 * Note that the root introspected type will be added to the catalog if it's also an {@link ObjectType}.
 *
 * @since 1.2.0, 1.1.7
 */
class CatalogTypeCollectorMetadataTypeVisitor extends MetadataTypeVisitor implements ReferenceResolver {

  private final LinkedHashMap<String, ObjectType> registeredTypes = new LinkedHashMap<>();
  private final List<String> index = new ArrayList<>();

  @Override
  public void visitObject(ObjectType type) {
    String id = createIdentifier(type);
    if (registeredTypes.containsKey(id)) {
      return;
    }
    registeredTypes.put(id, type);
    index.add(id);

    type.getOpenRestriction().ifPresent(t -> t.accept(this));
    type.getFields().forEach(field -> field.getValue().accept(this));
  }

  @Override
  public void visitArrayType(ArrayType arrayType) {
    arrayType.getType().accept(this);
  }

  @Override
  public void visitFunction(FunctionType functionType) {
    functionType.getParameters().forEach(p -> p.getType().accept(this));
    functionType.getReturnType().ifPresent(rt -> rt.accept(this));
  }

  @Override
  public void visitIntersection(IntersectionType intersectionType) {
    intersectionType.getTypes().forEach(t -> t.accept(this));
  }

  @Override
  public void visitUnion(UnionType unionType) {
    unionType.getTypes().forEach(t -> t.accept(this));
  }

  public Map<String, ObjectType> getCatalog() {
    Map<String, ObjectType> copy = new LinkedHashMap<>(registeredTypes.size());

    List<String> keys = new ArrayList<>(registeredTypes.keySet());
    for (String key : keys) {
      copy.put(referenceIndex(key), registeredTypes.get(key));
    }

    return copy;
  }

  private String referenceIndex(String key) {
    return "0x" + Integer.toHexString(index.indexOf(key) + 1);
  }

  @Override
  public Optional<String> getIdentifier(ObjectType type) {
    String id = createIdentifier(type);
    return registeredTypes.containsKey(id) ? of(referenceIndex(id)) : empty();
  }

  public String createIdentifier(ObjectType type) {
    Optional<String> typeId = getTypeId(type);
    return typeId.orElseGet(() -> String.valueOf(System.identityHashCode(type)));
  }

}
