/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.persistence;

import static java.util.Collections.unmodifiableList;
import static java.util.Optional.ofNullable;
import static org.mule.metadata.api.utils.MetadataTypeUtils.getTypeId;

import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.api.model.ObjectType;

import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

/**
 * Contains state which affects many serializers or type adapters involved in the process of marshalling a {@link MetadataType}
 *
 * @since 1.0
 */
public class SerializationContext {

  private final Map<String, ObjectType> objectTypes = new LinkedHashMap<>();

  /**
   * Acknowledges an {@link ObjectType} and returns a key which can be used as a reference to such type
   *
   * @param objectType an {@link ObjectType}
   * @return a key which acts as a reference to the given type
   */
  public String registerObjectType(ObjectType objectType) {
    final String typeId =
        getTypeId(objectType)
            .orElseThrow(() -> new IllegalArgumentException("Cannot doSerialize an object type without type id"));
    objectTypes.put(typeId, objectType);

    return typeId;
  }

  /**
   * Recovers an {@link ObjectType} which was registered through {@link #registerObjectType(ObjectType)}.
   *
   * @param registrationId the id that was obtained while registering the type
   * @return an {@link Optional} {@link ObjectType}
   */
  public Optional<ObjectType> getRegisteredObjectType(String registrationId) {
    return ofNullable(objectTypes.get(registrationId));
  }

  /**
   * @return An immutable {@link List} with all the {@link ObjectType} which were registered through
   *         {@link #registerObjectType(ObjectType)}
   */
  public List<ObjectType> getRegisteredObjectTypes() {
    return unmodifiableList(new ArrayList<>(objectTypes.values()));
  }
}
