/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.persistence.serializer;

import static org.mule.metadata.persistence.MetadataTypeConstants.RECURSION_FLAG;

import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.persistence.MetadataSerializingException;

import java.util.Stack;

import com.google.gson.stream.JsonWriter;

/**
 * {@link BaseTypeSerializer} specification that contains common logic to handle recursive scenarios.
 * <p>
 * All {@link TypeSerializer}s that can have recursive structures should inherit from this class.
 *
 * @param <T> the specific {@link MetadataType}
 * @since 1.2.0, 1.1.7
 */
public abstract class AbstractComplexTypeSerializer<T extends MetadataType> extends BaseTypeSerializer<T> {

  AbstractComplexTypeSerializer(String typeName) {
    super(typeName);
  }

  @Override
  public void serialize(JsonWriter writer, T type, Stack<MetadataType> typeStack) {
    if (typeStack.contains(type)) {
      // Recursion detected
      handleRecursion(writer, type, typeStack);
    } else {
      super.serialize(writer, type, typeStack);
    }
  }

  private void handleRecursion(JsonWriter writer, MetadataType type, Stack<MetadataType> typeStack) {
    final int indexOf = typeStack.indexOf(type);
    final int reference = typeStack.size() - indexOf;
    StringBuilder ref = new StringBuilder(RECURSION_FLAG);
    for (int i = 0; i < reference; i++) {
      if (i > 0) {
        ref.append("/");
      }
      ref.append("..");
    }
    try {
      writer.value(ref.toString());
    } catch (Exception e) {
      throw new MetadataSerializingException("recursive" + typeName + " MetadataType", e);
    }
  }
}
