/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.persistence.serializer;

import static org.mule.metadata.persistence.MetadataTypeConstants.ANY;
import static org.mule.metadata.persistence.MetadataTypeConstants.BINARY;
import static org.mule.metadata.persistence.MetadataTypeConstants.BOOLEAN;
import static org.mule.metadata.persistence.MetadataTypeConstants.DATE;
import static org.mule.metadata.persistence.MetadataTypeConstants.DATE_TIME;
import static org.mule.metadata.persistence.MetadataTypeConstants.NOTHING;
import static org.mule.metadata.persistence.MetadataTypeConstants.NULL;
import static org.mule.metadata.persistence.MetadataTypeConstants.NUMBER;
import static org.mule.metadata.persistence.MetadataTypeConstants.STRING;
import static org.mule.metadata.persistence.MetadataTypeConstants.TIME;
import static org.mule.metadata.persistence.MetadataTypeConstants.VOID;

import org.mule.metadata.api.model.AnyType;
import org.mule.metadata.api.model.ArrayType;
import org.mule.metadata.api.model.BinaryType;
import org.mule.metadata.api.model.BooleanType;
import org.mule.metadata.api.model.DateTimeType;
import org.mule.metadata.api.model.DateType;
import org.mule.metadata.api.model.FunctionType;
import org.mule.metadata.api.model.IntersectionType;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.api.model.NothingType;
import org.mule.metadata.api.model.NullType;
import org.mule.metadata.api.model.NumberType;
import org.mule.metadata.api.model.ObjectType;
import org.mule.metadata.api.model.StringType;
import org.mule.metadata.api.model.TimeType;
import org.mule.metadata.api.model.TupleType;
import org.mule.metadata.api.model.UnionType;
import org.mule.metadata.api.model.VoidType;
import org.mule.metadata.api.visitor.MetadataTypeVisitor;
import org.mule.metadata.persistence.ObjectTypeReferenceHandler;

import java.util.Stack;

import com.google.gson.stream.JsonWriter;

/**
 * {@link MetadataTypeVisitor} that serializes {@link MetadataType}s depending on it's nature.
 * <p>
 * It delegates on specific serializers for each type.
 *
 * @since 1.2.0, 1.1.7
 */
public class TypeSerializerVisitor extends MetadataTypeVisitor implements TypeSerializer<MetadataType> {

  private final JsonWriter writer;
  private final ObjectTypeReferenceHandler handler;
  private Stack<MetadataType> typeStack;

  private final SimpleTypeSerializer anyTypeSerializer;
  private final TypeReferenceSerializer referenceSerializer;
  private final ArrayTypeSerializer arrayTypeSerializer;
  private final SimpleTypeSerializer binaryTypeSerializer;
  private final SimpleTypeSerializer booleanTypeSerializer;
  private final SimpleTypeSerializer dateTimeSerializer;
  private final SimpleTypeSerializer dateSerializer;
  private final SimpleTypeSerializer nullTypeSerializer;
  private final SimpleTypeSerializer nothingTypeSerializer;
  private final SimpleTypeSerializer voidTypeSerializer;
  private final ObjectTypeSerializer objectTypeSerializer;
  private final SimpleTypeSerializer stringTypeSerializer;
  private final SimpleTypeSerializer timeTypeSerializer;
  private final TupleTypeSerializer tupleTypeSerializer;
  private final FunctionTypeSerializer functionTypeSerializer;
  private final UnionTypeSerializer unionTypeSerializer;
  private final IntersectionTypeSerializer intersectionTypeSerializer;

  public TypeSerializerVisitor(JsonWriter writer,
                               ObjectTypeReferenceHandler handler,
                               Stack<MetadataType> typeStack,
                               boolean formatOnReferences) {
    this.writer = writer;
    this.handler = handler;
    this.typeStack = typeStack;

    // Delegate Serializers
    this.anyTypeSerializer = new SimpleTypeSerializer(ANY);
    this.binaryTypeSerializer = new SimpleTypeSerializer(BINARY);
    this.booleanTypeSerializer = new SimpleTypeSerializer(BOOLEAN);
    this.dateTimeSerializer = new SimpleTypeSerializer(DATE_TIME);
    this.dateSerializer = new SimpleTypeSerializer(DATE);
    this.nullTypeSerializer = new SimpleTypeSerializer(NULL);
    this.nothingTypeSerializer = new SimpleTypeSerializer(NOTHING);
    this.voidTypeSerializer = new SimpleTypeSerializer(VOID);
    this.stringTypeSerializer = new SimpleTypeSerializer(STRING);
    this.timeTypeSerializer = new SimpleTypeSerializer(TIME);
    this.arrayTypeSerializer = new ArrayTypeSerializer(this);
    this.objectTypeSerializer = new ObjectTypeSerializer(this);
    this.tupleTypeSerializer = new TupleTypeSerializer(this);
    this.functionTypeSerializer = new FunctionTypeSerializer(this);
    this.unionTypeSerializer = new UnionTypeSerializer(this);
    this.intersectionTypeSerializer = new IntersectionTypeSerializer(this);
    this.referenceSerializer = new TypeReferenceSerializer(handler, formatOnReferences);
  }

  @Override
  public void serialize(JsonWriter writer, MetadataType type, Stack<MetadataType> typeStack) {
    this.typeStack = typeStack;
    type.accept(this);
  }

  @Override
  public void visitAnyType(AnyType anyType) {
    anyTypeSerializer.serialize(writer, anyType, typeStack);
  }

  @Override
  public void visitArrayType(ArrayType arrayType) {
    arrayTypeSerializer.serialize(writer, arrayType, typeStack);
  }

  @Override
  public void visitBinaryType(BinaryType binaryType) {
    binaryTypeSerializer.serialize(writer, binaryType, typeStack);
  }

  @Override
  public void visitBoolean(BooleanType booleanType) {
    booleanTypeSerializer.serialize(writer, booleanType, typeStack);
  }

  @Override
  public void visitDateTime(DateTimeType dateTimeType) {
    dateTimeSerializer.serialize(writer, dateTimeType, typeStack);
  }

  @Override
  public void visitDate(DateType dateType) {
    dateSerializer.serialize(writer, dateType, typeStack);
  }

  @Override
  public void visitNull(NullType nullType) {
    nullTypeSerializer.serialize(writer, nullType, typeStack);
  }

  @Override
  public void visitNothing(NothingType nothingType) {
    nothingTypeSerializer.serialize(writer, nothingType, typeStack);
  }

  @Override
  public void visitVoid(VoidType voidType) {
    voidTypeSerializer.serialize(writer, voidType, typeStack);
  }

  @Override
  public void visitNumber(NumberType numberType) {
    new SimpleTypeSerializer(NUMBER).serialize(writer, numberType, typeStack);
  }

  @Override
  public void visitObject(ObjectType objectType) {
    // we should check that the stack does not contain the current type, because in a recursive scenario
    // references should not be created but point to the element location in the tree.
    if (!typeStack.contains(objectType)) {
      if (handler.shouldWriteReference(objectType)) {
        referenceSerializer.serialize(writer, objectType, typeStack);
        return;
      }
    }
    objectTypeSerializer.serialize(writer, objectType, typeStack);
  }

  @Override
  public void visitString(StringType stringType) {
    stringTypeSerializer.serialize(writer, stringType, typeStack);
  }

  @Override
  public void visitTime(TimeType timeType) {
    timeTypeSerializer.serialize(writer, timeType, typeStack);
  }

  @Override
  public void visitTuple(TupleType tupleType) {
    tupleTypeSerializer.serialize(writer, tupleType, typeStack);
  }

  @Override
  public void visitFunction(FunctionType functionType) {
    functionTypeSerializer.serialize(writer, functionType, typeStack);
  }

  @Override
  public void visitUnion(UnionType unionType) {
    unionTypeSerializer.serialize(writer, unionType, typeStack);
  }

  @Override
  public void visitIntersection(IntersectionType intersectionType) {
    intersectionTypeSerializer.serialize(writer, intersectionType, typeStack);
  }
}
