/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.persistence;

import static org.mule.metadata.persistence.MetadataTypeConstants.FORMAT;
import static org.mule.metadata.persistence.MetadataTypeConstants.ID;
import static org.mule.metadata.persistence.MetadataTypeConstants.LABEL;
import static org.mule.metadata.persistence.MetadataTypeConstants.RECURSION_FLAG;
import static org.mule.metadata.persistence.MetadataTypeConstants.REF_FLAG;
import static org.mule.metadata.persistence.MetadataTypeConstants.TYPE;
import static org.mule.metadata.persistence.MetadataTypeConstants.VALID_MIME_TYPES;
import static org.mule.metadata.persistence.MetadataTypeConstants.commonMetadataFormats;

import org.mule.metadata.api.TypeLoader;
import org.mule.metadata.api.builder.BaseTypeBuilder;
import org.mule.metadata.api.builder.TypeBuilder;
import org.mule.metadata.api.model.MetadataFormat;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.persistence.deserializer.MetadataTypeDeserializerProvider;
import org.mule.metadata.persistence.deserializer.TypeDeserializer;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.Stack;

import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import com.google.gson.JsonSyntaxException;

/**
 * {@link TypeLoader} that creates instances of {@link MetadataType} from JSON representations generated by
 * {@link JsonMetadataTypeWriter}.
 * <p>
 * Instances of this class are not thread safe, so only one deserialization at a time can be made with each instance.
 *
 * @since 1.0
 */
public class JsonMetadataTypeLoader implements SerializedMetadataTypeLoader {

  private static final String MISSING_METADATA_FORMAT_ATTRIBUTE = "MetadataFormat object is malformed. '%s' property is required";
  private static final String UNSUPPORTED_METADATA_TYPE_FOUND = "Unsupported MetadataType '%s' found.";
  private final MetadataTypeDeserializerProvider typeDeserializerProvider;
  private final ObjectTypeReferenceHandler referenceHandler;
  private Stack<TypeBuilder> typeBuilderStack;
  private Stack<MetadataFormat> metadataFormatStack;

  /**
   * Creates a new instance which serializes all types explicitly without handling references
   */
  public JsonMetadataTypeLoader() {
    this(new NullObjectTypeReferenceHandler());
  }

  /**
   * Creates a new instance which handles type references through the given {@code referenceHandler}
   *
   * @param referenceHandler an {@link ObjectTypeReferenceHandler}
   */
  public JsonMetadataTypeLoader(ObjectTypeReferenceHandler referenceHandler) {
    this.referenceHandler = referenceHandler;
    this.typeDeserializerProvider = new MetadataTypeDeserializerProvider();
  }

  /**
   * Parses a {@link MetadataType} JSON representation and returns a new {@link MetadataType} instance.
   *
   * @param typeIdentifier {@link MetadataType} JSON representation as a {@link String}
   * @return {@link Optional} value of a {@link MetadataType}
   */
  @Override
  public Optional<MetadataType> load(String typeIdentifier) {
    try {
      final JsonElement metadataType = new JsonParser().parse(typeIdentifier);
      return load(metadataType);
    } catch (JsonSyntaxException e) {
      throw new MetadataDeserializingException(e);
    }
  }

  /**
   * Parses a {@link MetadataType} JSON representation and returns a new {@link MetadataType} instance.
   *
   * @param jsonElement {@link MetadataType} JSON representation as a {@link JsonElement}
   * @return {@link Optional} value of a {@link MetadataType}
   */
  @Override
  public Optional<MetadataType> load(JsonElement jsonElement) {
    return load(jsonElement, new Stack<>(), new Stack<>()).map(TypeBuilder::build);
  }

  public Optional<TypeBuilder> load(JsonElement jsonElement,
                                    Stack<TypeBuilder> builderStack,
                                    Stack<MetadataFormat> formatStack) {
    try {
      typeBuilderStack = builderStack;
      metadataFormatStack = formatStack;
      final JsonObject metadataType = jsonElement.getAsJsonObject();
      JsonElement typeElement = metadataType.get(TYPE);
      if (!typeElement.isJsonObject() && typeElement.getAsString().startsWith(REF_FLAG)) {
        return referenceHandler.readReference(typeElement.getAsString());
      }
      final JsonElement formatElement = metadataType.get(FORMAT);
      MetadataFormat format = buildMetadataFormat(formatElement);
      metadataFormatStack.push(format);
      final BaseTypeBuilder baseTypeBuilder = BaseTypeBuilder.create(format);
      return Optional.of(buildType(metadataType, baseTypeBuilder));
    } catch (Exception e) {
      throw new MetadataDeserializingException(e);
    }
  }

  public TypeBuilder buildType(JsonElement metadataTypeElement, BaseTypeBuilder baseBuilder) {
    typeBuilderStack.push(baseBuilder);
    TypeBuilder typeBuilder;
    if (metadataTypeElement.isJsonObject()) {
      JsonObject typeObject = metadataTypeElement.getAsJsonObject();

      final MetadataFormat currentMetadataFormat = updateMetadataFormat(typeObject, baseBuilder, metadataFormatStack.peek());
      metadataFormatStack.push(currentMetadataFormat);

      final String type = typeObject.get(TYPE).getAsString();

      final Optional<TypeDeserializer> typeDeserializer = typeDeserializerProvider.get(type);
      if (typeDeserializer.isPresent()) {
        typeBuilder = typeDeserializer.get().buildType(typeObject, baseBuilder, this);
      } else {
        typeBuilder = referenceHandler.readReference(type).orElse(null);
        if (typeBuilder == null) {
          throw new MetadataDeserializingException(String.format(UNSUPPORTED_METADATA_TYPE_FOUND, type));
        }
      }

      metadataFormatStack.pop();
    } else {
      final String typeReference = metadataTypeElement.getAsString();
      if (typeReference.startsWith(RECURSION_FLAG)) {
        typeBuilder = typeBuilderStack.get(getTypeIndex(typeReference));
      } else {
        throw new MetadataDeserializingException(String.format(UNSUPPORTED_METADATA_TYPE_FOUND, typeReference));
      }
    }

    typeBuilderStack.pop();

    return typeBuilder;
  }

  private int getTypeIndex(String typeReference) {
    return typeBuilderStack.size() - typeReference.split("/").length - 1;
  }

  private MetadataFormat updateMetadataFormat(JsonObject jsonElement, BaseTypeBuilder baseBuilder, MetadataFormat currentFormat) {
    MetadataFormat metadataFormat = currentFormat;
    final JsonObject type = jsonElement.getAsJsonObject();
    if (type.has(FORMAT)) {
      final JsonElement formatJsonElement = type.get(FORMAT);
      metadataFormat = buildMetadataFormat(formatJsonElement);
    }
    baseBuilder.withFormat(metadataFormat);
    return metadataFormat;
  }

  private MetadataFormat buildMetadataFormat(JsonElement formatElement) {
    MetadataFormat metadataFormat;
    if (formatElement.isJsonObject()) {
      List<String> validMimeTypesList = new ArrayList<>();
      String label = null;
      String id;
      final JsonObject formatObject = formatElement.getAsJsonObject();

      if (!formatObject.has(ID)) {
        throw new MetadataDeserializingException(String.format(MISSING_METADATA_FORMAT_ATTRIBUTE, ID));
      }
      id = formatObject.get(ID).getAsString();

      if (formatObject.has(LABEL)) {
        label = formatObject.get(LABEL).getAsString();
      }

      for (JsonElement validMimeTypes : formatObject.get(VALID_MIME_TYPES).getAsJsonArray()) {
        validMimeTypesList.add(validMimeTypes.getAsString());
      }

      metadataFormat = new MetadataFormat(label, id, validMimeTypesList.toArray(new String[validMimeTypesList.size()]));
    } else {
      final String formatId = formatElement.getAsString();
      final int formatIndex = commonMetadataFormats.indexOf(new MetadataFormat(LABEL, formatId));
      metadataFormat = commonMetadataFormats.get(formatIndex);
    }
    return metadataFormat;
  }
}
