/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.persistence.api.util;

import static java.util.Objects.requireNonNull;

import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.persistence.MetadataTypeGsonTypeAdapter;

import java.lang.reflect.Type;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonParseException;

/**
 * Serialization utils for {@link MetadataType}.
 *
 * @since 4.8
 */
public class SerializationUtils {

  private SerializationUtils() {
    // Nothing to do
  }

  /**
   * Serializes an {@link Object} containing {@link MetadataType} related stuff in {@code JSON} format.
   *
   * @param object {@link Object} to serialize.
   * @return a serialized JSON string.
   */
  public static String serializeMetadataType(Object object) {
    requireNonNull(object, "'object' to serialize cannot not be null");

    final Gson gson = new GsonBuilder().registerTypeAdapter(MetadataType.class, new MetadataTypeGsonTypeAdapter()).create();
    return gson.toJson(object);
  }

  /**
   * Deserializes a {@code JSON} string containing {@link MetadataType} related stuff into the given {@link Type}.
   *
   * @param json {@code JSON} string to deserialize.
   * @param type the {@link Type} of the object resulting of the deserialization.
   * @return an object of the given {@link Type}.
   * @param <T> type of the object resulting of the deserialization.
   */
  public static <T> T deserializeMetadataType(String json, Type type) {
    try {
      final Gson gson = new GsonBuilder().registerTypeAdapter(MetadataType.class, new MetadataTypeGsonTypeAdapter()).create();
      return gson.fromJson(json, type);
    } catch (JsonParseException e) {
      throw new IllegalArgumentException("The serialized content does not match the expected format", e);
    }
  }

}
