/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.persistence;

import org.mule.metadata.api.TypeWriter;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.persistence.reduced.ReducedJsonMetadataTypeWriter;

import java.io.IOException;
import java.io.StringWriter;
import java.util.Stack;

import com.google.gson.stream.JsonWriter;

/**
 * Base class for {@link MetadataTypeWriter}s implementations that serialize {@link MetadataType} objects into a readable and
 * processable JSON file, that can be deserialize again into a {@link MetadataType} using a {@link SerializedMetadataTypeLoader}.
 * <p>
 * This class is not Thread-safe.
 *
 * @since 1.2.0, 1.1.7
 */
public abstract class AbstractJsonMetadataTypeWriter implements TypeWriter, MetadataTypeWriter {

  private static final String INDENT_TAB = "  ";

  protected JsonWriter writer;
  protected Stack<MetadataType> typeStack = new Stack<>();
  protected boolean prettyPrint;

  /**
   * Given a {@link MetadataType}, serializes it into JSON format, the result of the serialization will be the output of the
   * operation as a {@link String}
   *
   * @param structure {@link MetadataType} to doSerialize.
   * @throws MetadataSerializingException in any error case
   */
  @Override
  public String toString(MetadataType structure) {
    try {
      StringWriter out = new StringWriter();
      writer = new JsonWriter(out);
      if (prettyPrint) {
        writer.setIndent(INDENT_TAB);
      }
      write(structure, writer);
      return out.toString();
    } catch (IOException e) {
      throw new MetadataSerializingException("the given MetadataType", e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void write(MetadataType metadataType, JsonWriter writer) throws IOException {
    this.writer = writer;
    write(metadataType);
    this.typeStack = new Stack<>();
  }

  /**
   * Given a {@link MetadataType}, serializes it into a JSON format.
   * <p>
   * It is expected that a proper writer is already set to this object.
   *
   * @param type {@link MetadataType} to doSerialize.
   * @throws IOException if any error occurred when writing the JSON file.
   */
  protected abstract void write(MetadataType type) throws IOException;

  /**
   * Changes the value of PrettyPrint property of the {@link ReducedJsonMetadataTypeWriter}. If it is enabled, when
   * {@link #toString(MetadataType)} is executed the output {@link String} will be printed in a human readable format, otherwise,
   * by default, the JSON will be printed in a compact and more performable format.
   *
   * @param prettyPrint boolean value indicating if the writer should pretty print the JSON
   * @return the configured {@link TypeWriter} instance
   */
  public AbstractJsonMetadataTypeWriter setPrettyPrint(boolean prettyPrint) {
    this.prettyPrint = prettyPrint;
    return this;
  }
}
