/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.persistence;


import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.TypeAdapter;
import org.mule.metadata.api.model.MetadataType;

import java.lang.reflect.Type;

/**
 * {@link Gson} {@link TypeAdapter<MetadataType>} that helps to serialize and deserialize {@link MetadataType} objects.
 * <p>
 * To use it, register it into {@link Gson} use the {@link GsonBuilder} and invoking
 * {@link GsonBuilder#registerTypeAdapter(Type type, Object typeAdapter)}, with {@link MetadataType} class as the {@code type}
 * parameter and an instance of {@link MetadataTypeGsonTypeAdapter} for the {@code typeAdapter} parameter.
 *
 * @since 1.0
 */
public final class MetadataTypeGsonTypeAdapter extends BaseMetadataTypeGsonTypeAdapter {

  /**
   * Creates a new instance which serializes all types explicitly without handling references
   */
  public MetadataTypeGsonTypeAdapter() {
    this(new NullObjectTypeReferenceHandler());
  }

  /**
   * Creates a new instance which serializes all types explicitly without handling references
   */
  public MetadataTypeGsonTypeAdapter(boolean allowNullInput) {
    this(new NullObjectTypeReferenceHandler(), allowNullInput);
  }

  /**
   * Creates a new instance which handles type references through the given {@code referenceHandler}
   *
   * @param referenceHandler an {@link ObjectTypeReferenceHandler}
   */
  public MetadataTypeGsonTypeAdapter(ObjectTypeReferenceHandler referenceHandler) {
    super(new JsonMetadataTypeWriter(referenceHandler), new JsonMetadataTypeLoader(referenceHandler));
  }

  public MetadataTypeGsonTypeAdapter(ObjectTypeReferenceHandler referenceHandler, boolean allowNullInput) {
    super(new JsonMetadataTypeWriter(referenceHandler, allowNullInput),
          new JsonMetadataTypeLoader(referenceHandler));
    this.setAllowNullInput(allowNullInput);
  }

  /**
   * Creates a new instance which serializes all types with the provided writer and loader.
   * <p>
   * {@link TypeAdapter} implementation is done in this base class, concrete implementations should provide a
   * {@link MetadataTypeWriter} and a {@link SerializedMetadataTypeLoader} implementations that actually know how to
   * serialize/deserialize the {@link MetadataType} in some JSON structure.
   *
   * @since 1.8.0
   */
  public MetadataTypeGsonTypeAdapter(MetadataTypeWriter writer, SerializedMetadataTypeLoader loader) {
    super(writer, loader);
  }
}
