/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.raml.api;

import org.mule.metadata.api.TypeLoader;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.raml.internal.RamlTypeLoader;
import org.mule.metadata.raml.internal.TypeDeclarationTypeLoader;
import org.mule.metadata.xml.SchemaCollector;
import org.mule.metadata.xml.XmlTypeLoader;
import org.mule.metadata.xml.utils.XmlSchemaUtils;
import org.raml.v2.api.model.v10.RamlFragment;
import org.raml.v2.api.model.v10.datamodel.TypeDeclaration;

import javax.xml.namespace.QName;
import java.io.File;
import java.util.Arrays;
import java.util.Optional;

/**
 RAML {@link TypeLoader} that takes the internal {@link TypeDeclaration} and transforms it into a {@link MetadataType} through
 * {@link TypeDeclaration#toXmlSchema()} in conjunction with {@link XmlTypeLoader}.
 * <p/>
 * Supported fragments so far: {@link RamlFragment#DataType} and {@link RamlFragment#Library}, any other will fail while loading t
 * he RAML file
 *
 * @since 1.0
 */
public class XmlRamlTypeLoader implements TypeLoader {

  private final RamlTypeLoader ramlTypeLoader;

  /**
   * Constructs a RAML type loader
   *
   * @param ramlFile file pointing to the RAML resource
   */
  public XmlRamlTypeLoader(File ramlFile) {
    this.ramlTypeLoader = new RamlTypeLoader(ramlFile, createXmlTypeDeclarationTypeLoader());
  }

  /**
   * Constructs a RAML type loader
   *
   * @param content content of the RAML resource
   * @param ramlLocation pointer to the RAML location
   */
  public XmlRamlTypeLoader(String content, String ramlLocation) {
    this.ramlTypeLoader = new RamlTypeLoader(content, ramlLocation, createXmlTypeDeclarationTypeLoader());
  }

  private TypeDeclarationTypeLoader createXmlTypeDeclarationTypeLoader() {
    return (typeDeclaration, typeIdentifier, typeAlias) -> {
      final String schemaContent = typeDeclaration.toXmlSchema();
      final XmlTypeLoader xmlTypeLoader = new XmlTypeLoader(SchemaCollector.getInstance().addSchema("", schemaContent));
      //as the RAML parser will generate an schema with targetNamespace, we will look for the only existing QName in it
      final Optional<QName> xmlSchemaRootElementName =
          XmlSchemaUtils.getXmlSchemaRootElementName(Arrays.asList(schemaContent), "");
      return xmlSchemaRootElementName.flatMap(qName -> xmlTypeLoader.load(xmlSchemaRootElementName.get(), typeAlias));
    };
  }

  @Override
  public Optional<MetadataType> load(String typeIdentifier) {
    return load(typeIdentifier, null);
  }

  @Override
  public Optional<MetadataType> load(String typeIdentifier, String typeAlias) {
    return ramlTypeLoader.load(typeIdentifier, typeAlias);
  }
}
