/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.raml.internal;

import org.mule.metadata.api.TypeLoader;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.raml.internal.fragments.AbstractTypeDeclarationLoader;
import org.mule.metadata.raml.internal.fragments.DataTypeTypeDeclarationLoader;
import org.mule.metadata.raml.internal.fragments.LibraryTypeDeclarationLoader;
import org.mule.metadata.raml.internal.util.RamlLoaderUtils;
import org.raml.v2.api.RamlModelResult;
import org.raml.v2.api.model.v10.RamlFragment;

import java.io.File;
import java.util.Optional;

/**
 * RAML {@link TypeLoader} that takes the type from it and transforms it into a {@link MetadataType}.
 * <p/>
 * Supported fragments so far: {@link RamlFragment#DataType} and {@link RamlFragment#Library}, any other will fail while loading t
 * he RAML file
 *
 * @since 1.0
 */
public class RamlTypeLoader implements TypeLoader {

  private final AbstractTypeDeclarationLoader fragmentTypeLoader;
  private final TypeDeclarationTypeLoader typeDeclarationTypeLoader;

  public RamlTypeLoader(File ramlFile, TypeDeclarationTypeLoader typeDeclarationTypeLoader) {
    final RamlModelResult ramlModelResult = RamlLoaderUtils.buildApiOrFail(ramlFile);
    this.fragmentTypeLoader = createDelegate(ramlModelResult, ramlFile.getPath());
    this.typeDeclarationTypeLoader = typeDeclarationTypeLoader;
  }

  public RamlTypeLoader(String content, String ramlLocation, TypeDeclarationTypeLoader typeDeclarationTypeLoader) {
    final RamlModelResult ramlModelResult = RamlLoaderUtils.buildApiOrFail(content, ramlLocation);
    this.fragmentTypeLoader = createDelegate(ramlModelResult, ramlLocation);
    this.typeDeclarationTypeLoader = typeDeclarationTypeLoader;
  }

  private AbstractTypeDeclarationLoader createDelegate(RamlModelResult ramlModelResult, String ramlLocation) {
    AbstractTypeDeclarationLoader result;
    switch (ramlModelResult.getFragment()) {
      case DataType:
        result = new DataTypeTypeDeclarationLoader(ramlModelResult);
        break;
      case Library:
        result = new LibraryTypeDeclarationLoader(ramlModelResult);
        break;
      default:
        throw new IllegalArgumentException(String
            .format("Unsupported [%s] fragment in RAML file '%s'.",
                    ramlModelResult.getFragment() == null ? "" : ramlModelResult.getFragment(), ramlLocation));
    }
    return result;
  }

  @Override
  public Optional<MetadataType> load(String typeIdentifier) {
    return load(typeIdentifier, null);
  }

  @Override
  public Optional<MetadataType> load(String typeIdentifier, String typeAlias) {
    return fragmentTypeLoader.load(typeIdentifier)
        .flatMap(typeDeclaration -> typeDeclarationTypeLoader.load(typeDeclaration, typeIdentifier, typeAlias));
  }
}
