/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.raml.internal.fragments.handler;

import org.mule.metadata.api.builder.BaseTypeBuilder;
import org.mule.metadata.api.builder.StringTypeBuilder;
import org.mule.metadata.api.builder.TypeBuilder;
import org.raml.v2.api.model.v10.datamodel.StringTypeDeclaration;
import org.raml.v2.api.model.v10.datamodel.TypeDeclaration;

import java.util.List;
import java.util.Optional;

public class StringTypeDeclarationHandler extends BaseTypeDeclarationHandler {

  @Override
  public boolean handles(TypeDeclaration typeDeclaration) {
    return typeDeclaration instanceof StringTypeDeclaration;
  }

  @Override
  public TypeBuilder<?> handle(TypeDeclaration typeDeclaration, BaseTypeBuilder root,
                               TypeDeclarationHandlerManager typeDeclarationHandlerManager,
                               ParsingContext parsingContext) {
    StringTypeDeclaration stringTypeDeclaration = (StringTypeDeclaration) typeDeclaration;

    final StringTypeBuilder stringTypeBuilder = root.stringType();

    final List<String> enumValues = stringTypeDeclaration.enumValues();
    if (enumValues != null && enumValues.size() > 0) {
      stringTypeBuilder.enumOf(enumValues.toArray(new String[enumValues.size()]));
    }

    final Optional<Integer> minLength = Optional.ofNullable(stringTypeDeclaration.minLength());
    final Optional<Integer> maxLength = Optional.ofNullable(stringTypeDeclaration.maxLength());
    if (minLength.isPresent() && maxLength.isPresent()) {
      stringTypeBuilder.boundary(minLength.get(), maxLength.get());
    } else if (maxLength.isPresent()) {
      stringTypeBuilder.length(maxLength.get());
    }

    final Optional<String> pattern = Optional.ofNullable(stringTypeDeclaration.pattern());
    if (pattern.isPresent()) {
      stringTypeBuilder.pattern(pattern.get());
    }

    handleAnnotations(stringTypeBuilder, stringTypeDeclaration.annotations());

    final Optional<String> defaultValueOptional = Optional.ofNullable(stringTypeDeclaration.defaultValue());
    if (defaultValueOptional.isPresent()) {
      stringTypeBuilder.defaultValue(defaultValueOptional.get());
    }

    return stringTypeBuilder;
  }
}
