/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.raml.api.csv;

import static org.mule.metadata.api.model.MetadataFormat.CSV;
import org.mule.metadata.api.TypeLoader;
import org.mule.metadata.api.model.ArrayType;
import org.mule.metadata.api.model.BooleanType;
import org.mule.metadata.api.model.DateTimeType;
import org.mule.metadata.api.model.DateType;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.api.model.NumberType;
import org.mule.metadata.api.model.ObjectType;
import org.mule.metadata.api.model.StringType;
import org.mule.metadata.raml.internal.CustomHandlingTypeDeclarationTypeLoader;
import org.mule.metadata.raml.internal.RamlTypeLoader;

import java.io.File;
import java.util.Arrays;
import java.util.List;
import java.util.Optional;

/**
 * Loader of a CSV from a RAML file with a few restrictions. The obtained {@link MetadataType} from the RAML must be of only some
 * valid field types, see {@link #VALID_FIELD_TYPES}
 *
 * @since 1.0
 */
public class CsvTypeLoader implements TypeLoader {

  public static final List<Class<? extends MetadataType>> VALID_FIELD_TYPES =
      Arrays.asList(StringType.class, NumberType.class, BooleanType.class,
                    DateType.class, DateTimeType.class);

  private final RamlTypeLoader ramlTypeLoader;

  /**
   * Constructs a CSV type loader from a RAML document
   *
   * @param ramlFile file pointing to the RAML resource
   */
  public CsvTypeLoader(File ramlFile) {
    ramlTypeLoader = new RamlTypeLoader(ramlFile, new CustomHandlingTypeDeclarationTypeLoader(CSV));
  }

  /**
   * Constructs a CSV type loader from a RAML document
   *
   * @param content      content of the RAML resource
   * @param ramlLocation pointer to the RAML location
   */
  public CsvTypeLoader(String content, String ramlLocation) {
    ramlTypeLoader = new RamlTypeLoader(content, ramlLocation, new CustomHandlingTypeDeclarationTypeLoader(CSV));
  }

  @Override
  public Optional<MetadataType> load(String typeIdentifier) {
    return load(typeIdentifier, null);
  }

  @Override
  public Optional<MetadataType> load(String typeIdentifier, String typeAlias) {
    return ramlTypeLoader.load(typeIdentifier, typeAlias).map(metadataType -> {
      validate(metadataType);
      return metadataType;
    });
  }

  private void validateFieldType(MetadataType metadataType) {
    for (Class<? extends MetadataType> validFieldMetadataTypeClass : VALID_FIELD_TYPES) {
      if (validFieldMetadataTypeClass.isAssignableFrom(metadataType.getClass())) {
        return;
      }
    }
    throw new RuntimeException("Invalid RAML type for CSV format. Only primitive type fields are allowed.");
  }

  private void validate(MetadataType metadataType) {
    if (metadataType instanceof ArrayType) {
      ArrayType arrayType = (ArrayType) metadataType;
      final MetadataType elementMetadataType = arrayType.getType();
      if (elementMetadataType instanceof ObjectType) {
        ObjectType objectType = (ObjectType) elementMetadataType;
        objectType.getFields().forEach(objectFieldType -> {
          validateFieldType(objectFieldType.getValue());
        });
      } else {
        throw new RuntimeException(
                                   "Invalid RAML type for CSV format. Only arrays of objects with primitive type fields are allowed.");
      }
    } else {
      throw new RuntimeException(
                                 "Invalid RAML type for CSV format. Only arrays of objects with primitive type fields are allowed.");
    }
  }
}
