/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.xml;

import static java.util.Collections.unmodifiableMap;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.LinkedHashMap;
import java.util.Map;

public class SchemaCollector {

  private final Map<String, InputStream> schemas;

  private SchemaCollector() {
    this.schemas = new LinkedHashMap<>();
  }

  public static SchemaCollector getInstance() {
    return new SchemaCollector();
  }

  public SchemaCollector addSchema(File schema) {
    try {
      schemas.put(schema.toURI().toString(), new FileInputStream(schema));
    } catch (FileNotFoundException e) {
      throw new RuntimeException("Error while adding schema file [" + schema.getName() + "]", e);
    }
    return this;
  }

  public SchemaCollector addSchema(URL schemaUrl) {
    try {
      schemas.put(schemaUrl.toString(), schemaUrl.openStream());
    } catch (IOException e) {
      throw new RuntimeException("Error while adding schema of URL [" + schemaUrl + "]", e);
    }
    return this;
  }

  public SchemaCollector addSchema(String name, String schemaContent) {
    schemas.put(name, new ByteArrayInputStream(schemaContent.getBytes()));
    return this;
  }

  public SchemaCollector addSchema(String schemaUrl) {
    try {
      addSchema(new URL(schemaUrl));
    } catch (MalformedURLException e) {
      throw new RuntimeException("The specified URL [" + schemaUrl + "] is invalid", e);
    }
    return this;
  }

  public SchemaCollector addSchema(String name, InputStream schemaContent) {
    schemas.put(name, schemaContent);
    return this;
  }

  public Map<String, InputStream> collect() {
    return unmodifiableMap(schemas);
  }
}
