/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.xml;

import org.mule.metadata.api.annotation.ExampleAnnotation;
import org.mule.metadata.xml.utils.SchemaHelper;
import org.mule.metadata.xml.utils.XmlSchemaUtils;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.StringReader;
import java.nio.file.Files;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

import javax.xml.namespace.QName;

import org.apache.commons.lang3.StringEscapeUtils;
import org.apache.xerces.dom.DOMInputImpl;
import org.apache.xerces.impl.xs.XSImplementationImpl;
import org.apache.xerces.impl.xs.XSLoaderImpl;
import org.apache.xerces.impl.xs.util.LSInputListImpl;
import org.apache.xerces.xs.XSLoader;
import org.apache.xerces.xs.XSModel;
import org.w3c.dom.DOMConfiguration;
import org.w3c.dom.DOMErrorHandler;

public class ModelFactory {

  private XSModel model;
  private Optional<QName> rootElementName;
  private Optional<ExampleAnnotation> example;

  private ModelFactory(XSModel model, ExampleAnnotation example, QName rootElementName) {
    this.model = model;
    this.example = Optional.ofNullable(example);
    this.rootElementName = Optional.ofNullable(rootElementName);
  }

  public XSModel getModel() {
    return model;
  }

  public Optional<QName> getRootElementName() {
    return rootElementName;
  }

  public Optional<ExampleAnnotation> getExample() {
    return example;
  }

  public static ModelFactory fromExample(File exampleFile) {
    try {
      return fromExample(Files.readAllLines(exampleFile.toPath()).stream().collect(Collectors.joining()));
    } catch (IOException e) {
      throw new RuntimeException(e);
    }
  }

  public static ModelFactory fromExample(String exampleXML) {

    try {
      final XSLoader schemaLoader = initializeXSLoader();
      List<String> schemas = SchemaHelper.generateXSD(exampleXML);

      final DOMInputImpl[] domInputs = schemas.stream()
          .map((schema) -> new DOMInputImpl(null, null, null, new StringReader(schema), "UTF-8")).toArray(DOMInputImpl[]::new);
      final XSModel model = schemaLoader.loadInputList(new LSInputListImpl(domInputs, domInputs.length));
      final Optional<QName> rootElementName = XmlSchemaUtils.getXmlSchemaRootElementName(schemas, exampleXML);
      return new ModelFactory(model, new ExampleAnnotation(StringEscapeUtils.escapeXml11(exampleXML)),
                              rootElementName.orElse(null));
    } catch (Exception e) {
      throw new RuntimeException(e);
    }
  }

  public static ModelFactory fromSchemas(Map<String, InputStream> schemasMap) {
    try {
      final XSLoader schemaLoader = initializeXSLoader();
      DOMConfiguration myLoader = (XSLoaderImpl) schemaLoader;
      List<String> errors = new ArrayList<>();
      myLoader.setParameter("error-handler", (DOMErrorHandler) error -> {
        errors.add(error.getMessage());
        return false;
      });
      final DOMInputImpl[] domInputs = schemasMap.entrySet()
          .stream()
          .map((schema) -> new DOMInputImpl(null, schema.getKey(), null, schema.getValue(), "UTF-8"))
          .toArray(DOMInputImpl[]::new);
      final XSModel model = schemaLoader.loadInputList(new LSInputListImpl(domInputs, domInputs.length));
      if (model == null) {
        throw new RuntimeException("Failed while trying to load schema errors "
            + errors.stream().reduce("", (value, acc) -> value + ", " + acc));
      }
      return new ModelFactory(model, null, null);
    } catch (Exception e) {
      throw new RuntimeException(e);
    }
  }

  private static XSLoader initializeXSLoader() throws ClassNotFoundException, InstantiationException, IllegalAccessException {
    final XSImplementationImpl impl = new XSImplementationImpl();
    return impl.createXSLoader(null);
  }
}
