/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.xml.api.utils;

import static org.apache.xmlbeans.XmlBeans.compileXmlBeans;

import org.mule.metadata.internal.utils.StringUtils;
import org.mule.metadata.xml.api.XmlTypeLoader;

import java.io.IOException;
import java.io.Reader;
import java.io.StringReader;
import java.util.List;
import java.util.Optional;

import javax.xml.namespace.QName;

import org.apache.xmlbeans.SchemaGlobalElement;
import org.apache.xmlbeans.SchemaTypeLoader;
import org.apache.xmlbeans.SchemaTypeSystem;
import org.apache.xmlbeans.XmlException;
import org.apache.xmlbeans.XmlObject;
import org.apache.xmlbeans.XmlOptions;
import org.apache.xmlbeans.impl.schema.BuiltinSchemaTypeSystem;
import org.apache.xmlbeans.impl.schema.SchemaTypeLoaderImpl;
import org.w3c.dom.Node;

public class XmlSchemaUtils {


  /**
   * @param schemas
   * @param example
   * @return
   */
  public static Optional<QName> getXmlSchemaRootElementName(final List<String> schemas, final String example) {
    try {
      SchemaTypeSystem schemaTypeSystem = getSchemaTypeSystem(schemas);
      final SchemaGlobalElement[] globalElements = schemaTypeSystem.globalElements();

      if (globalElements.length == 1) {
        return Optional.ofNullable(globalElements[0].getName());
      } else if (globalElements.length > 0 && StringUtils.isNotEmpty(example)) {
        try (Reader exampleReader = new StringReader(example)) {
          return Optional.ofNullable(getRootElementName(exampleReader));
        }
      } else if (globalElements.length > 0) {
        // There are more than one possible global element. Choosing the first option.;
        return Optional.ofNullable(globalElements[0].getName());
      } else {
        return Optional.empty();
      }
    } catch (Exception e) {
      throw new RuntimeException("Fail to get XML root element name", e);
    }
  }

  public static QName getRootElementName(final Reader xmlDocument) throws IOException {
    XmlObject xmlInstance;
    try {
      xmlInstance = XmlObject.Factory.parse(xmlDocument);
    } catch (XmlException e) {
      throw new IOException(e);
    }
    Node rootNode = xmlInstance.getDomNode().getFirstChild();
    return new QName(rootNode.getNamespaceURI(), rootNode.getLocalName());

  }

  private static SchemaTypeSystem getSchemaTypeSystem(List<String> schemas) throws Exception {
    final XmlOptions options = new XmlOptions();
    options.setCompileNoUpaRule();
    options.setCompileNoValidation();
    options.setCompileDownloadUrls();

    /* Load the schema */
    final XmlObject[] schemaRepresentation = new XmlObject[schemas.size()];
    final SchemaTypeLoader contextTypeLoader =
        SchemaTypeLoaderImpl.build(new SchemaTypeLoader[] {BuiltinSchemaTypeSystem.get()}, null,
                                   XmlTypeLoader.class.getClassLoader());

    int i = 0;
    for (String schemaStream : schemas) {
      XmlObject schemaObject = contextTypeLoader.parse(schemaStream, null, null);
      schemaRepresentation[i] = schemaObject;
      i++;
    }
    return compileXmlBeans(null, null, schemaRepresentation, null, contextTypeLoader, null, options);
  }
}
