/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.xml.internal;

import org.mule.apache.xerces.dom.DOMInputImpl;
import org.w3c.dom.ls.LSInput;
import org.w3c.dom.ls.LSResourceResolver;

import java.util.List;
import java.util.Map;

import static java.util.Objects.requireNonNull;

public class ResourceResolverFactory {

  public static final String LOAD_EXTERNAL_RESOURCES = "mule.xml.loadExternalResources";

  private ResourceResolverFactory() {

  }

  /**
   * Create LSResourceResolver that will be used to find preloaded XML schemas in the schemaByTargetNamespace map. If the
   * environment variable "mule.xml.loadExternalResources" is set to true, resources not found in the map will be resolved on the
   * fly, otherwise NullPointerException will be thrown.
   *
   * @param schemaByTargetNamespace map containing the preloaded XSDs
   * @return resource resolver.
   */
  public static LSResourceResolver create(Map<String, List<DOMInputImpl>> schemaByTargetNamespace) {
    LSResourceResolver resourceResolver = new ResourceResolver(schemaByTargetNamespace);
    if (!Boolean.valueOf(System.getProperty(LOAD_EXTERNAL_RESOURCES, "true"))) {
      resourceResolver = new ResourceResolverRequiredDecorator(resourceResolver);
    }
    return resourceResolver;
  }

  private static class ResourceResolverRequiredDecorator implements LSResourceResolver {

    LSResourceResolver resourceResolver;

    public ResourceResolverRequiredDecorator(LSResourceResolver resourceResolver) {
      this.resourceResolver = resourceResolver;
    }

    @Override
    public LSInput resolveResource(String type, String namespaceURI, String publicId, String systemId, String baseURI) {
      return requireNonNull(resourceResolver.resolveResource(type, namespaceURI, publicId, systemId, baseURI));
    }
  }

}
