/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.xml.api.utils;

import org.mule.metadata.xml.api.schema.SchemaFetcher;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.IOException;
import java.io.StringWriter;
import java.io.Writer;
import java.net.MalformedURLException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Optional;

import org.apache.xmlbeans.XmlException;
import org.apache.xmlbeans.XmlObject;
import org.apache.xmlbeans.XmlOptions;
import org.apache.xmlbeans.impl.inst2xsd.Inst2Xsd;
import org.apache.xmlbeans.impl.inst2xsd.Inst2XsdOptions;
import org.apache.xmlbeans.impl.xb.xsdschema.SchemaDocument;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

public class SchemaHelper {

  public static Collection<XmlDoc> getDocumentation(String annotationContent) {

    final List<XmlDoc> result = new ArrayList<>();
    final DocumentBuilderFactory dbFactory = DocumentBuilderFactory.newInstance();
    try {
      final DocumentBuilder dBuilder = dbFactory.newDocumentBuilder();
      final Document doc = dBuilder.parse(new ByteArrayInputStream(annotationContent.getBytes("UTF-8")));
      final NodeList children = doc.getDocumentElement().getChildNodes();
      for (int i = 0; i < children.getLength(); i++) {
        final Node child = children.item(i);
        if (isDocumentationTag(child)) {
          final String textContent = child.getTextContent();
          final NamedNodeMap attributes = child.getAttributes();
          final Optional<String> lang =
              Optional.ofNullable(attributes.getNamedItem("xml:lang")).map((attr) -> attr.getNodeValue());
          result.add(new XmlDoc(lang, textContent.trim()));
        }
      }
    } catch (Exception e) {
      // todo: Ignore this exception???
    }
    return result;
  }

  public static boolean isDocumentationTag(Node child) {
    return child instanceof Element
        && getLocalName((Element) child).endsWith("documentation");
  }

  public static String getLocalName(Element child) {
    final String tagName = child.getTagName();
    return tagName.contains(":") ? tagName.split(":")[1] : tagName;
  }

  public static List<SchemaFetcher.XsdDependency> getDependencies(File schema) {
    List<SchemaFetcher.XsdDependency> dependencies = Collections.emptyList();
    try {
      dependencies = new SchemaFetcher().fetchAll(schema.toURI().toURL());
    } catch (MalformedURLException e) {
      throw new RuntimeException(e);
    }
    return dependencies;
  }

  public static List<String> generateXSD(String xmlExample) throws IOException {
    try {
      return generateXSD(XmlObject.Factory.parse(xmlExample));
    } catch (XmlException e) {
      throw new IOException(e);
    }
  }

  public static List<String> generateXSD(File xmlFile) throws IOException {
    try {
      return generateXSD(XmlObject.Factory.parse(xmlFile));
    } catch (XmlException e) {
      throw new IOException(e);
    }
  }

  private static List<String> generateXSD(XmlObject xmlInstance) throws IOException {
    List<String> schemas = new ArrayList<>();
    Inst2XsdOptions options = new Inst2XsdOptions();
    SchemaDocument[] inst2xsd = Inst2Xsd.inst2xsd(new XmlObject[] {xmlInstance}, options);

    for (SchemaDocument schemaDocument : inst2xsd) {
      Writer writer = new StringWriter();
      XmlOptions xmlOptions = new XmlOptions();
      schemaDocument.save(writer, xmlOptions.setSavePrettyPrint());
      schemas.add(writer.toString());
    }
    return schemas;
  }

  public static class XmlDoc {

    private String content;
    private Optional<String> lang;

    public XmlDoc(Optional<String> lang, String content) {
      this.lang = lang;
      this.content = content;
    }

    public Optional<String> getLang() {
      return lang;
    }

    public String getContent() {
      return content;
    }
  }
}
