/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.xml.api;

import static java.lang.Boolean.parseBoolean;
import static org.mule.apache.xerces.impl.Constants.DISALLOW_DOCTYPE_DECL_FEATURE;
import static org.mule.apache.xerces.impl.Constants.DOM_ERROR_HANDLER;
import static org.mule.apache.xerces.impl.Constants.HONOUR_ALL_SCHEMALOCATIONS_FEATURE;
import static org.mule.apache.xerces.impl.Constants.XERCES_FEATURE_PREFIX;
import static org.mule.apache.xerces.impl.xs.XMLSchemaLoader.ENTITY_RESOLVER;
import static org.mule.metadata.xml.api.ModelFactoryFromExampleSupport.fixForMultipleSchemas;

import org.mule.apache.xerces.dom.DOMInputImpl;
import org.mule.apache.xerces.impl.xs.SchemaSymbols;
import org.mule.apache.xerces.impl.xs.XSImplementationImpl;
import org.mule.apache.xerces.impl.xs.XSLoaderImpl;
import org.mule.apache.xerces.impl.xs.opti.SchemaDOMParser;
import org.mule.apache.xerces.impl.xs.opti.SchemaParsingConfig;
import org.mule.apache.xerces.impl.xs.util.LSInputListImpl;
import org.mule.apache.xerces.util.DOMEntityResolverWrapper;
import org.mule.apache.xerces.util.DOMUtil;
import org.mule.apache.xerces.xni.parser.XMLInputSource;
import org.mule.apache.xerces.xs.XSLoader;
import org.mule.apache.xerces.xs.XSModel;
import org.mule.metadata.api.annotation.ExampleAnnotation;
import org.mule.metadata.internal.utils.StringUtils;
import org.mule.metadata.xml.api.utils.SchemaHelper;
import org.mule.metadata.xml.api.utils.XmlSchemaUtils;
import org.mule.metadata.xml.internal.ResourceResolver;

import javax.xml.namespace.QName;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.StringReader;
import java.nio.file.Files;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringEscapeUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.DOMErrorHandler;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

public class ModelFactory {

  private static final String EXPAND_ENTITIES_PROPERTY = "mule.xml.expandInternalEntities";
  private static final Logger LOGGER = LoggerFactory.getLogger(ModelFactory.class);

  private XSModel model;
  private Optional<QName> rootElementName;
  private Optional<ExampleAnnotation> example;

  private ModelFactory(XSModel model, ExampleAnnotation example, QName rootElementName) {
    this.model = model;
    this.example = Optional.ofNullable(example);
    this.rootElementName = Optional.ofNullable(rootElementName);
  }

  public static ModelFactory fromExample(File exampleFile) {
    try {
      return fromExample(Files.readAllLines(exampleFile.toPath()).stream().collect(Collectors.joining()));
    } catch (IOException e) {
      throw new RuntimeException(e);
    }
  }

  public static ModelFactory fromExample(String exampleXML) {
    final XSLoader schemaLoader;
    List<String> schemas;
    try {
      schemaLoader = initializeXSLoader();
      schemas = SchemaHelper.generateXSD(exampleXML);
    } catch (Exception e) {
      throw new RuntimeException("Failed to generated XSD schemas.", e);
    }

    if (schemas.size() > 1) {
      try {
        schemas = fixForMultipleSchemas(schemaLoader, schemas);

      } catch (Exception e) {
        throw new RuntimeException("Failed to configure XSD loader.", e);
      }
    }

    try {
      final DOMInputImpl[] domInputs = schemas.stream()
          .map((schema) -> new DOMInputImpl(null, null, null, new StringReader(schema), "UTF-8"))
          .toArray(DOMInputImpl[]::new);
      final XSModel model = schemaLoader.loadInputList(new LSInputListImpl(domInputs, domInputs.length));
      final Optional<QName> rootElementName = XmlSchemaUtils.getXmlSchemaRootElementName(schemas, exampleXML);
      return new ModelFactory(model, new ExampleAnnotation(StringEscapeUtils.escapeXml11(exampleXML)),
                              rootElementName.orElse(null));
    } catch (Exception e) {
      throw new RuntimeException("Failed to generate metadatatype from XSD.", e);
    }
  }

  public static ModelFactory fromSchemas(Map<String, InputStream> schemasMap) {
    try {
      final XSLoaderImpl schemaLoader = (XSLoaderImpl) initializeXSLoader();
      final Map<String, List<DOMInputImpl>> schemaByTargetNamespace = getSchemasByTargetNamespace(schemasMap);
      ResourceResolver resourceResolver = new ResourceResolver(schemaByTargetNamespace);
      schemaLoader.setParameter(ENTITY_RESOLVER, new DOMEntityResolverWrapper(resourceResolver));
      List<String> errors = new ArrayList<>();
      String expandEntitiesValue = System.getProperty(EXPAND_ENTITIES_PROPERTY, "false");
      schemaLoader.setParameter(XERCES_FEATURE_PREFIX + DISALLOW_DOCTYPE_DECL_FEATURE, !parseBoolean(expandEntitiesValue));
      schemaLoader.setParameter(XERCES_FEATURE_PREFIX + HONOUR_ALL_SCHEMALOCATIONS_FEATURE, true);
      schemaLoader.setParameter(DOM_ERROR_HANDLER, (DOMErrorHandler) error -> {
        errors.add(error.getMessage());
        return false;
      });
      final DOMInputImpl[] domInputs = resourceResolver.getSchemaArray();
      final XSModel model = schemaLoader.loadInputList(new LSInputListImpl(domInputs, domInputs.length));
      if (model == null) {
        throw new RuntimeException("Failed while trying to load schema errors "
            + errors.stream().reduce("", (value, acc) -> value + ", " + acc));
      }
      return new ModelFactory(model, null, null);
    } catch (Exception e) {
      throw new RuntimeException(e);
    }
  }

  private static Map<String, List<DOMInputImpl>> getSchemasByTargetNamespace(Map<String, InputStream> schemasMap)
      throws IOException {
    SchemaDOMParser schemaDOMParser = new SchemaDOMParser(new SchemaParsingConfig());
    Map<String, List<DOMInputImpl>> schemaByTargetNamespace = new LinkedHashMap<>();
    int i = 0;
    for (Map.Entry<String, InputStream> schema : schemasMap.entrySet()) {
      String schemaString = IOUtils.toString(schema.getValue(), "UTF-8");
      DOMInputImpl domInput = new DOMInputImpl(null, schema.getKey(), null, schemaString, "UTF-8");
      String targetNamespace = getTargetNamespace(schema.getKey(), schemaString, schemaDOMParser);
      String key = StringUtils.isNotEmpty(targetNamespace) ? targetNamespace : "NO_NAMESPACE_" + i++;
      if (!schemaByTargetNamespace.containsKey(key)) {
        schemaByTargetNamespace.put(key, new LinkedList<DOMInputImpl>());
      }
      schemaByTargetNamespace.get(key).add(domInput);
    }
    return schemaByTargetNamespace;
  }

  private static String getTargetNamespace(String systemId, String schema, SchemaDOMParser domParser) {
    try {
      domParser.parse(new XMLInputSource(null, systemId, null, new StringReader(schema), "UTF-8"));
    } catch (IOException e) {
      LOGGER.error("Error getting namespace from schema: " + e.getMessage(), e);
      return null;
    }
    Document document = domParser.getDocument();
    Element element = document != null ? DOMUtil.getRoot(document) : null;
    return DOMUtil.getAttrValue(element, SchemaSymbols.ATT_TARGETNAMESPACE);
  }

  private static XSLoader initializeXSLoader() {
    final XSImplementationImpl impl = new XSImplementationImpl();
    return impl.createXSLoader(null);
  }

  XSModel getModel() {
    return model;
  }

  public Optional<QName> getRootElementName() {
    return rootElementName;
  }

  public Optional<ExampleAnnotation> getExample() {
    return example;
  }

}
