/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.xml.api;

import org.mule.apache.xerces.dom.DOMInputImpl;
import org.mule.apache.xerces.xs.XSLoader;
import org.w3c.dom.Document;
import org.w3c.dom.ls.LSInput;
import org.w3c.dom.ls.LSResourceResolver;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.Result;
import javax.xml.transform.Source;
import javax.xml.transform.Templates;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerConfigurationException;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.stream.StreamResult;
import javax.xml.transform.stream.StreamSource;
import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathConstants;
import javax.xml.xpath.XPathExpression;
import javax.xml.xpath.XPathExpressionException;
import javax.xml.xpath.XPathFactory;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.StringReader;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

import static java.lang.String.format;

public class ModelFactoryFromExampleSupport {

  static List<String> fixForMultipleSchemas(XSLoader schemaLoader, List<String> schemas) throws TransformerException {
    List<String> targetNamespaces = new ArrayList<>();
    schemas.forEach(schema -> {
      try {
        targetNamespaces.add(findTargetNamespace(schema).orElse(""));
      } catch (ParserConfigurationException | XPathExpressionException | IOException | SAXException e) {
        throw new RuntimeException("Failed resolving example metadata type", e);
      }
    });

    Map<String, String> schemasByNamespaceUri = new HashMap<>();
    List<String> tmpSchemas = new ArrayList<>();
    for (int i = 0; i < schemas.size(); i++) {
      final String currentTargetNamespace = targetNamespaces.get(i);
      final String schema = schemas.get(i);
      final String fixedSchema = fixSchemaAddingImports(
                                                        schema,
                                                        targetNamespaces
                                                            .stream()
                                                            .filter(targetNamespace -> !targetNamespace
                                                                .equals(currentTargetNamespace))
                                                            .collect(Collectors.toList()));
      tmpSchemas.add(fixedSchema);
      schemasByNamespaceUri.put(currentTargetNamespace, fixedSchema);
    }
    schemaLoader.getConfig().setParameter("resource-resolver", new LSResourceResolver() {

      @Override
      public LSInput resolveResource(String type, String namespaceURI, String publicId, String systemId, String baseURI) {
        return Optional
            .ofNullable(schemasByNamespaceUri.get(Optional.ofNullable(namespaceURI).orElse("")))
            .map(schema -> new DOMInputImpl(publicId, systemId, baseURI, new StringReader(schema), "UTF-8")).orElse(null);
      }
    });
    return tmpSchemas;
  }

  private static String fixSchemaAddingImports(String schema, List<String> targetNamespaces)
      throws TransformerException {
    ByteArrayOutputStream resultOutputStream = new ByteArrayOutputStream();
    Templates template = buildSchemaAddingTransform(targetNamespaces);
    Transformer transformer = template.newTransformer();
    Source streamSource = new StreamSource(new StringReader(schema));
    Result streamResult = new StreamResult(resultOutputStream);
    transformer.transform(streamSource, streamResult);
    return resultOutputStream.toString();
  }

  private static Templates buildSchemaAddingTransform(List<String> targetNamespaces) throws TransformerConfigurationException {
    StringBuilder stringBuilder = new StringBuilder();
    targetNamespaces.forEach(targetNamespace -> {
      stringBuilder
          .append(targetNamespace.isEmpty() ? "<xs:import xmlns:xs=\"http://www.w3.org/2001/XMLSchema\"/>"
              : format("<xs:import namespace=\"%s\" xmlns:xs=\"http://www.w3.org/2001/XMLSchema\"/>", targetNamespace));
    });
    String importsSection = stringBuilder.toString();

    TransformerFactory transformerFactory = TransformerFactory.newInstance();
    return transformerFactory.newTemplates(new StreamSource(new StringReader(
                                                                             format("<?xml version = \"1.0\" encoding = \"UTF-8\"?>\n"
                                                                                 + "<xsl:stylesheet version = \"1.0\" xmlns:xsl = \"http://www.w3.org/1999/XSL/Transform\">\n"
                                                                                 + "  <xsl:template match=\"node()|@*\">\n"
                                                                                 + "    <xsl:copy>\n"
                                                                                 + "      <xsl:apply-templates select=\"node()|@*\"/>\n"
                                                                                 + "    </xsl:copy>\n"
                                                                                 + "  </xsl:template>\n"
                                                                                 + "\n"
                                                                                 + "  <xsl:template match=\"/xs:schema\" xmlns:xs=\"http://www.w3.org/2001/XMLSchema\">\n"
                                                                                 + "    <xsl:copy>\n"
                                                                                 + "      <xsl:apply-templates select=\"@*\"/>\n"
                                                                                 + "      %s\n"
                                                                                 + "      <xsl:apply-templates select=\"node()\"/>\n"
                                                                                 + "    </xsl:copy>\n"
                                                                                 + "  </xsl:template>\n"
                                                                                 + "\n"
                                                                                 + "</xsl:stylesheet>", importsSection))));
  }


  private static Optional<String> findTargetNamespace(String schema) throws ParserConfigurationException,
      XPathExpressionException, IOException, SAXException {
    InputSource source = new InputSource(new StringReader(schema));
    String path = "string(/schema/@targetNamespace)";
    DocumentBuilderFactory documentBuilderFactory = DocumentBuilderFactory.newInstance();
    DocumentBuilder documentBuilder = documentBuilderFactory.newDocumentBuilder();
    Document doc = documentBuilder.parse(source);
    XPathFactory xPathfactory = XPathFactory.newInstance();
    XPath xpath = xPathfactory.newXPath();
    XPathExpression xPathExpression = xpath.compile(path);
    return Optional.ofNullable((String) xPathExpression.evaluate(doc, XPathConstants.STRING));
  }
}
