/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.xml.api;

import static java.util.Collections.unmodifiableMap;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.URL;
import java.nio.charset.Charset;
import java.util.LinkedHashMap;
import java.util.Map;

import org.apache.commons.io.input.BOMInputStream;

public class SchemaCollector {

  private final Map<String, InputStream> schemas;
  private final String charset;

  private SchemaCollector(String charset) {
    this.charset = charset;
    this.schemas = new LinkedHashMap<>();
  }

  public static SchemaCollector getInstance() {
    return new SchemaCollector(Charset.defaultCharset().toString());
  }

  public static SchemaCollector getInstance(String charset) {
    return new SchemaCollector(charset);
  }

  public SchemaCollector addSchema(File schema) {
    try {
      schemas.put(schema.toURI().toString(), wrapInBOMInputStream(new FileInputStream(schema)));
    } catch (FileNotFoundException e) {
      throw new RuntimeException("Error while adding schema file [" + schema.getName() + "]", e);
    }
    return this;
  }

  public SchemaCollector addSchema(URL schemaUrl) {
    try {
      schemas.put(schemaUrl.toString(), wrapInBOMInputStream(schemaUrl.openStream()));
    } catch (IOException e) {
      throw new RuntimeException("Error while adding schema of URL [" + schemaUrl + "]", e);
    }
    return this;
  }

  public SchemaCollector addSchema(String name, String schemaContent) {
    schemas.put(name, wrapInBOMInputStream(new ByteArrayInputStream(schemaContent.getBytes(Charset.forName(charset)))));
    return this;
  }

  public SchemaCollector addSchema(String schemaUrl) {
    try {
      addSchema(new URL(schemaUrl));
    } catch (MalformedURLException e) {
      throw new RuntimeException("The specified URL [" + schemaUrl + "] is invalid", e);
    }
    return this;
  }

  public SchemaCollector addSchema(String name, InputStream schemaContent) {
    schemas.put(name, wrapInBOMInputStream(schemaContent));
    return this;
  }

  public Map<String, InputStream> collect() {
    return unmodifiableMap(schemas);
  }

  /**
   * Wraps the input strema in a {@link BOMInputStream} to make sure that we can handle this scenario.
   *
   * Otherwise XML Parsers with fail with error "Content not valid in prolog."
   * @param streamToWrap the InputStream to be wrapped.
   * @return a {@link BOMInputStream} that can handle xmls with and without BOM markers
   */
  private InputStream wrapInBOMInputStream(InputStream streamToWrap) {
    return new BOMInputStream(streamToWrap);
  }
}
