/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.xml.api.handler;

import static org.mule.apache.xerces.xs.XSSimpleTypeDefinition.FACET_LENGTH;
import static org.mule.apache.xerces.xs.XSSimpleTypeDefinition.FACET_MAXLENGTH;
import static org.mule.apache.xerces.xs.XSSimpleTypeDefinition.FACET_MINLENGTH;
import static org.mule.apache.xerces.xs.XSSimpleTypeDefinition.FACET_PATTERN;
import static java.util.Optional.ofNullable;
import static org.mule.metadata.xml.internal.utils.SchemaHelper.getFacet;

import org.mule.metadata.api.builder.BaseTypeBuilder;
import org.mule.metadata.api.builder.StringTypeBuilder;
import org.mule.metadata.api.builder.TypeBuilder;
import org.mule.metadata.xml.api.utils.XmlConstants;

import org.mule.apache.xerces.impl.dv.xs.XSSimpleTypeDecl;
import org.mule.apache.xerces.xs.StringList;
import org.mule.apache.xerces.xs.XSMultiValueFacet;
import org.mule.apache.xerces.xs.XSObject;
import org.mule.metadata.xml.internal.utils.SchemaHelper;

import java.util.Arrays;
import java.util.List;
import java.util.Optional;

import javax.xml.namespace.QName;

public class StringHandler implements XmlHandler {

  public static final List<QName> STRING_TYPES =
      Arrays.asList(XmlConstants.XSD_STRING, XmlConstants.XSD_ANY, XmlConstants.XSD_NORMALIZEDSTRING);

  @Override
  public boolean handles(QName schema) {
    return STRING_TYPES.contains(schema);
  }

  @Override
  public TypeBuilder<?> handle(XSSimpleTypeDecl element, Optional<String> defaultValue, BaseTypeBuilder builder) {
    final StringTypeBuilder stringType = builder.stringType();
    final StringList enumeration = element.getLexicalEnumeration();
    if (!enumeration.isEmpty()) {
      final String[] values = (String[]) enumeration.toArray(new String[enumeration.size()]);
      stringType.enumOf(values);
    } else {
      if (defaultValue.isPresent() && !defaultValue.get().isEmpty()) {
        stringType.defaultValue(defaultValue.get());
      }

      // Pattern
      final XSObject patternFacetObject = getFacet(element, FACET_PATTERN);
      if (patternFacetObject instanceof XSMultiValueFacet) {
        XSMultiValueFacet patternFacet = (XSMultiValueFacet) patternFacetObject;
        final StringList lexicalFacetValues = patternFacet.getLexicalFacetValues();
        if (lexicalFacetValues != null && !lexicalFacetValues.isEmpty()) {
          stringType.pattern(lexicalFacetValues.item(0));
        }
      }

      // Boundary
      final Optional<Integer> minLength = Optional.ofNullable(getFacet(element, FACET_MINLENGTH))
          .map(SchemaHelper::getIntegerValue);
      final Optional<Integer> maxLength = Optional.ofNullable(getFacet(element, FACET_MAXLENGTH))
          .map(SchemaHelper::getIntegerValue);
      if (minLength.isPresent() || maxLength.isPresent()) {
        stringType.boundary(minLength.orElse(null), maxLength.orElse(null));
      }

      // Length
      final Optional<Integer> length = ofNullable(getFacet(element, FACET_LENGTH))
          .map(SchemaHelper::getIntegerValue);
      length.ifPresent(stringType::length);

    }
    return stringType;
  }
}
