/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.xml.api;

import static java.lang.Boolean.parseBoolean;
import static java.nio.file.Files.readAllLines;
import static org.apache.commons.lang3.StringEscapeUtils.escapeXml11;
import static org.mule.apache.xerces.impl.Constants.DISALLOW_DOCTYPE_DECL_FEATURE;
import static org.mule.apache.xerces.impl.Constants.DOM_ERROR_HANDLER;
import static org.mule.apache.xerces.impl.Constants.HONOUR_ALL_SCHEMALOCATIONS_FEATURE;
import static org.mule.apache.xerces.impl.Constants.XERCES_FEATURE_PREFIX;
import static org.mule.apache.xerces.impl.XMLEntityManager.expandSystemId;
import static org.mule.apache.xerces.impl.xs.XMLSchemaLoader.ENTITY_RESOLVER;
import static org.mule.metadata.internal.utils.StringUtils.isNotEmpty;
import static org.mule.metadata.xml.api.ModelFactoryFromExampleSupport.fixForMultipleSchemas;
import static org.mule.metadata.xml.api.utils.SchemaHelper.generateXSD;
import static org.mule.metadata.xml.api.utils.XmlSchemaUtils.getXmlSchemaRootElementName;
import static org.mule.metadata.xml.internal.ResourceResolver.NO_NAMESPACE;
import org.mule.apache.xerces.dom.DOMInputImpl;
import org.mule.apache.xerces.impl.xs.SchemaSymbols;
import org.mule.apache.xerces.impl.xs.XSImplementationImpl;
import org.mule.apache.xerces.impl.xs.XSLoaderImpl;
import org.mule.apache.xerces.impl.xs.opti.SchemaDOMParser;
import org.mule.apache.xerces.impl.xs.opti.SchemaParsingConfig;
import org.mule.apache.xerces.impl.xs.util.LSInputListImpl;
import org.mule.apache.xerces.util.DOMEntityResolverWrapper;
import org.mule.apache.xerces.util.DOMUtil;
import org.mule.apache.xerces.util.URI;
import org.mule.apache.xerces.xni.parser.XMLInputSource;
import org.mule.apache.xerces.xs.LSInputList;
import org.mule.apache.xerces.xs.XSLoader;
import org.mule.apache.xerces.xs.XSModel;
import org.mule.metadata.api.annotation.ExampleAnnotation;
import org.mule.metadata.xml.internal.ResourceResolverFactory;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.StringReader;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

import javax.xml.namespace.QName;

import org.apache.commons.io.IOUtils;
import org.apache.commons.io.input.BOMInputStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.DOMErrorHandler;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.ls.LSInput;
import org.w3c.dom.ls.LSResourceResolver;

public class ModelFactory {

  public static final String EXPAND_ENTITIES_PROPERTY = "mule.xml.expandInternalEntities";
  private static final Logger LOGGER = LoggerFactory.getLogger(ModelFactory.class);

  private XSModel model;
  private Optional<QName> rootElementName;
  private Optional<ExampleAnnotation> example;

  private ModelFactory(XSModel model, ExampleAnnotation example, QName rootElementName) {
    this.model = model;
    this.example = Optional.ofNullable(example);
    this.rootElementName = Optional.ofNullable(rootElementName);
  }

  public static ModelFactory fromExample(File exampleFile) {
    try {
      return fromExample(readAllLines(exampleFile.toPath()).stream().collect(Collectors.joining()));
    } catch (IOException e) {
      throw new RuntimeException(e);
    }
  }

  public static ModelFactory fromExample(String exampleXML) {
    final XSLoader schemaLoader;
    List<String> schemas;
    try {
      schemaLoader = initializeXSLoader();
      schemas = generateXSD(exampleXML);
    } catch (Exception e) {
      throw new RuntimeException("Failed to generated XSD schemas.", e);
    }

    if (schemas.size() > 1) {
      try {
        schemas = fixForMultipleSchemas(schemaLoader, schemas);

      } catch (Exception e) {
        throw new RuntimeException("Failed to configure XSD loader.", e);
      }
    }

    try {
      final DOMInputImpl[] domInputs = schemas.stream()
          .map((schema) -> new DOMInputImpl(null, null, null, new StringReader(schema), "UTF-8"))
          .toArray(DOMInputImpl[]::new);
      final XSModel model = schemaLoader.loadInputList(new LSInputListImpl(domInputs, domInputs.length));
      final Optional<QName> rootElementName = getXmlSchemaRootElementName(schemas, exampleXML);
      String exampleWithNoBom = exampleXML;
      try (InputStream exampleStream = new ByteArrayInputStream(exampleXML.getBytes());
          BOMInputStream stream = new BOMInputStream(exampleStream)) {
        exampleWithNoBom = IOUtils.toString(stream, StandardCharsets.UTF_8);
      } catch (Exception ex) {
        LOGGER.debug("Unexpected error removing BOM from example", ex);
      }

      return new ModelFactory(model, new ExampleAnnotation(escapeXml11(exampleWithNoBom)),
                              rootElementName.orElse(null));
    } catch (Exception e) {
      throw new RuntimeException("Failed to generate metadatatype from XSD.", e);
    }
  }

  public static ModelFactory fromSchemas(Map<String, InputStream> schemasMap) {
    try {
      final XSLoaderImpl schemaLoader = (XSLoaderImpl) initializeXSLoader();
      final Map<String, List<DOMInputImpl>> schemaByTargetNamespace = getSchemasByTargetNamespace(schemasMap);
      LSResourceResolver resourceResolver = ResourceResolverFactory.create(schemaByTargetNamespace);
      schemaLoader.setParameter(ENTITY_RESOLVER, new DOMEntityResolverWrapper(resourceResolver));
      schemaLoader.setParameter(XERCES_FEATURE_PREFIX + DISALLOW_DOCTYPE_DECL_FEATURE, isDisallowDoctypeDeclarations());
      schemaLoader.setParameter(XERCES_FEATURE_PREFIX + HONOUR_ALL_SCHEMALOCATIONS_FEATURE, true);
      List<String> errors = new ArrayList<>();
      schemaLoader.setParameter(DOM_ERROR_HANDLER, (DOMErrorHandler) error -> {
        errors.add(error.getMessage());
        return false;
      });
      final XSModel model = schemaLoader.loadInputList(getLSInputList(schemaByTargetNamespace));
      if (model == null) {
        throw new RuntimeException("Failed while trying to load schema errors "
            + errors.stream().reduce("", (value, acc) -> value + ", " + acc));
      }
      return new ModelFactory(model, null, null);
    } catch (Exception e) {
      throw new RuntimeException(e);
    }
  }

  private static boolean isDisallowDoctypeDeclarations() {
    String expandEntitiesValue = System.getProperty(EXPAND_ENTITIES_PROPERTY, "false");
    return !parseBoolean(expandEntitiesValue);
  }

  private static Map<String, List<DOMInputImpl>> getSchemasByTargetNamespace(Map<String, InputStream> schemasMap)
      throws IOException {
    SchemaDOMParser schemaDOMParser = new SchemaDOMParser(new SchemaParsingConfig());
    schemaDOMParser.setFeature(XERCES_FEATURE_PREFIX + DISALLOW_DOCTYPE_DECL_FEATURE, isDisallowDoctypeDeclarations());

    Map<String, List<DOMInputImpl>> schemaByTargetNamespace = new LinkedHashMap<>();
    for (Map.Entry<String, InputStream> schema : schemasMap.entrySet()) {
      String schemaString = IOUtils.toString(schema.getValue(), StandardCharsets.UTF_8);
      DOMInputImpl domInput = new DOMInputImpl(null, getSystemId(schema.getKey()), null, schemaString, "UTF-8");
      String targetNamespace = getTargetNamespace(schema.getKey(), schemaString, schemaDOMParser);
      String key = isNotEmpty(targetNamespace) ? targetNamespace : NO_NAMESPACE;
      if (!schemaByTargetNamespace.containsKey(key)) {
        schemaByTargetNamespace.put(key, new LinkedList<>());
      }
      schemaByTargetNamespace.get(key).add(domInput);
    }
    return schemaByTargetNamespace;
  }

  private static String getSystemId(String systemId) {
    try {
      return expandSystemId(systemId, null, false);
    } catch (URI.MalformedURIException e) {
      return systemId;
    }
  }

  private static String getTargetNamespace(String systemId, String schema, SchemaDOMParser domParser) {
    try {
      domParser.parse(new XMLInputSource(null, systemId, null, new StringReader(schema), "UTF-8"));
    } catch (IOException e) {
      LOGGER.error("Error getting namespace from schema: " + e.getMessage(), e);
      return null;
    }
    Document document = domParser.getDocument();
    Element element = document != null ? DOMUtil.getRoot(document) : null;
    return DOMUtil.getAttrValue(element, SchemaSymbols.ATT_TARGETNAMESPACE);
  }

  private static XSLoader initializeXSLoader() {
    final XSImplementationImpl impl = new XSImplementationImpl();
    return impl.createXSLoader(null);
  }

  private static LSInputList getLSInputList(Map<String, List<DOMInputImpl>> schemaByTargetNamespace) {
    List<DOMInputImpl> domInputs = new LinkedList<>();
    schemaByTargetNamespace.values().forEach(domInputs::addAll);
    LSInput[] inputs = domInputs.toArray(new LSInput[0]);
    return new LSInputListImpl(inputs, inputs.length);
  }

  XSModel getModel() {
    return model;
  }

  public Optional<QName> getRootElementName() {
    return rootElementName;
  }

  public Optional<ExampleAnnotation> getExample() {
    return example;
  }

}
