/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.xml.api.schema;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathConstants;
import javax.xml.xpath.XPathExpression;
import javax.xml.xpath.XPathExpressionException;
import javax.xml.xpath.XPathFactory;
import java.io.IOException;
import java.io.InputStream;
import java.io.StringReader;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Scanner;

import org.apache.commons.io.IOUtils;
import org.apache.commons.io.input.BOMInputStream;
import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.EntityResolver;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

/**
 * This class provide the functionality of fetching all the XSD dependencies of a XSd file using import or include references.
 */
public class SchemaFetcher {

  private static Map<String, ContentFetcher> contentFetchers = new HashMap<>();

  static {
    contentFetchers.put("http", new HTTPContentFetcher());
    contentFetchers.put("file", new FileContentFetcher());
  }

  public List<XsdDependency> fetchAll(URL... urls) {
    return new SchemaFetcherRequest().fetchAll(urls);
  }

  /**
   * Represents an import or include dependency in a XSD file
   */
  public static class XsdDependency {

    private URL parent;

    private URL dependency;

    public XsdDependency(URL parent, URL dependency) {
      this.parent = parent;
      this.dependency = dependency;
    }

    public URL getParent() {
      return parent;
    }

    public URL getDependency() {
      return dependency;
    }
  }

  private static class SchemaFetcherRequest {

    private Map<String, Schema> schemas = new HashMap<>();

    public List<XsdDependency> fetchAll(URL... urls) {
      List<XsdDependency> dependencies = new ArrayList<>();
      for (URL url : urls) {
        Schema schema = new Schema(url);
        dependencies.addAll(schema.fetchDependencies());
      }
      return dependencies;
    }

    private class Schema {

      private static final String FILE = "file";
      private static final String HTTP = "http";

      private static final String CURRENT_DIRECTORY = ".";
      private static final String BACKWARD_DIRECTORY = "..";

      private static final String SCHEMA_LOCATION = "schemaLocation";

      private URL url;

      private Map<String, Schema> includesAndImports = new HashMap<>();

      public Schema(URL url) {
        this.url = url;
      }

      public URL getUrl() {
        return url;
      }

      public List<XsdDependency> fetchDependencies() {
        List<SchemaFetcher.XsdDependency> dependencies = new ArrayList<>();
        ContentFetcher contentFetcher = SchemaFetcher.contentFetchers.get(url.getProtocol());
        Optional<String> maybeContent = contentFetcher != null ? contentFetcher.fetchContent(url) : Optional.empty();
        if (maybeContent.isPresent()) {
          InputSource source = new InputSource(new StringReader(maybeContent.get()));

          List<Node> includesAndImportsNodes =
              getXpathNodes(source, "/*[local-name()='schema']/*[local-name()='include' or local-name()='import']");

          for (Node element : includesAndImportsNodes) {
            Node schemaLocationAttribute = element.getAttributes().getNamedItem(SCHEMA_LOCATION);
            if (schemaLocationAttribute == null) {
              continue;
            }
            String schemaLocation = schemaLocationAttribute.getNodeValue();

            Optional<URL> maybeUrl = buildUrl(url, schemaLocation);

            if (maybeUrl.isPresent()) {
              URL dependency = maybeUrl.get();
              Schema schema = new Schema(dependency);
              includesAndImports.put(schemaLocation, schema);
              dependencies.add(new XsdDependency(url, dependency));
            }
          }

          schemas.put(url.toString(), this);

          for (Schema schemaToIncludeOrImport : includesAndImports.values()) {
            if (!schemas.containsKey(schemaToIncludeOrImport.getUrl().toString())) {
              dependencies.addAll(schemaToIncludeOrImport.fetchDependencies());
            }
          }
        }
        return dependencies;
      }

      private List<Node> getXpathNodes(InputSource source, String path) {
        try (InputStream sourceAsStream =
            IOUtils.toInputStream(IOUtils.toString(source.getCharacterStream()), StandardCharsets.UTF_8);
            BOMInputStream sanitizedInputStream = new BOMInputStream(sourceAsStream)) {
          DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
          DocumentBuilder builder = factory.newDocumentBuilder();
          builder.setEntityResolver(new NoOpEntityResolver());

          InputSource bomInputSource = new InputSource(sanitizedInputStream);
          Document doc = builder.parse(bomInputSource);

          XPathFactory xPathfactory = XPathFactory.newInstance();
          XPath xpath = xPathfactory.newXPath();
          XPathExpression expr = xpath.compile(path);

          NodeList nodeList = (NodeList) expr.evaluate(doc, XPathConstants.NODESET);
          int size = nodeList.getLength();

          List<Node> nodes = new ArrayList<>(size);

          for (int i = 0; i < size; i++) {
            nodes.add(nodeList.item(i));
          }

          return nodes;
        } catch (XPathExpressionException | ParserConfigurationException | IOException | SAXException e) {
          throw new RuntimeException(e);
        }
      }

      private Optional<URL> buildUrl(URL parent, String schemaLocation) {
        try {
          if (schemaLocation.startsWith(HTTP) || schemaLocation.startsWith(FILE)) {
            return Optional.of(new URL(schemaLocation));
          }

          // Relative URL
          URI parentUri = parent.toURI().getPath().endsWith("/") ? parent.toURI().resolve(BACKWARD_DIRECTORY)
              : parent.toURI().resolve(CURRENT_DIRECTORY);
          URL url = new URL(parentUri.toURL().toString() + schemaLocation);
          return Optional.of(new URL(url.toURI().normalize().toString()));
        } catch (MalformedURLException | URISyntaxException e) {
          throw new RuntimeException(e);
        }
      }
    }
  }

  private interface ContentFetcher {

    Optional<String> fetchContent(URL url);
  }

  private static class FileContentFetcher implements ContentFetcher {

    private static final String DELIMITER = "\\Z";

    @Override
    public Optional<String> fetchContent(URL url) {
      try (Scanner scanner = new Scanner(url.openStream(), StandardCharsets.UTF_8.name())) {
        return Optional.of(scanner.useDelimiter(DELIMITER).next());
      } catch (IOException e) {
        throw new RuntimeException(e);
      }
    }
  }

  private static class HTTPContentFetcher implements ContentFetcher {

    private static final String DELIMITER = "\\A";

    @Override
    public Optional<String> fetchContent(URL url) {
      try (Scanner scanner = new Scanner(url.openStream(), StandardCharsets.UTF_8.name())) {
        return Optional.of(scanner.useDelimiter(DELIMITER).next());
      } catch (IOException e) {
        throw new RuntimeException(e);
      }
    }
  }

  private static class NoOpEntityResolver implements EntityResolver {

    @Override
    public InputSource resolveEntity(String publicId, String systemId)
        throws SAXException, IOException {
      return new InputSource(new StringReader(""));
    }
  }
}
