/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.xml.internal;

import org.mule.apache.xerces.dom.DOMInputImpl;

import java.util.List;
import java.util.Map;

import org.w3c.dom.ls.LSInput;
import org.w3c.dom.ls.LSResourceResolver;

import static org.apache.commons.lang3.StringUtils.isEmpty;
import static org.mule.metadata.internal.utils.StringUtils.isNotEmpty;
import static org.mule.apache.xerces.impl.XMLEntityManager.expandSystemId;

public class ResourceResolver implements LSResourceResolver {

  public static final String NO_NAMESPACE = "NO_NAMESPACE";

  private Map<String, List<DOMInputImpl>> schemaByTargetNamespace;

  public ResourceResolver(Map<String, List<DOMInputImpl>> schemaByTargetNamespace) {
    this.schemaByTargetNamespace = schemaByTargetNamespace;
  }

  @Override
  public LSInput resolveResource(String type, String namespaceURI, String publicId, String systemId, String baseURI) {
    String resolvedURI = getAbsoluteUri(systemId, baseURI);
    if (isNotEmpty(namespaceURI) && schemaByTargetNamespace.containsKey(namespaceURI)) {
      return resolveResourceForNamespace(schemaByTargetNamespace.get(namespaceURI), resolvedURI);
    } else if (isEmpty(namespaceURI) && isNotEmpty(systemId)) {
      return resolveResourceForNamespace(schemaByTargetNamespace.get(NO_NAMESPACE), resolvedURI);
    }
    return null;
  }

  private String getAbsoluteUri(String systemId, String baseURI) {
    try {
      return baseURI == null ? systemId : expandSystemId(systemId, baseURI, false);
    } catch (Exception e) {
      return null;
    }
  }

  private LSInput resolveResourceForNamespace(List<DOMInputImpl> domInputs, String systemId) {
    if (domInputs != null) {
      for (DOMInputImpl domInput : domInputs) {
        if (systemId == null || systemId.equals(domInput.getSystemId())) {
          return domInput;
        }
      }
    }
    return null;
  }
}
