/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.metrics.exporter.config.impl;

import static org.mule.runtime.container.api.MuleFoldersUtil.getConfFolder;
import static org.mule.runtime.metrics.exporter.config.api.OpenTelemetryMeterExporterConfigurationProperties.MULE_OPEN_TELEMETRY_METER_EXPORTER_CONFIGURATION_FILE_PATH;

import static java.lang.String.format;
import static java.lang.System.getProperty;
import static java.nio.file.Paths.get;
import static java.util.Optional.empty;

import org.mule.runtime.config.api.properties.ConfigurationPropertiesResolver;
import org.mule.runtime.config.internal.model.dsl.config.DefaultConfigurationPropertiesResolver;
import org.mule.runtime.config.internal.model.dsl.config.SystemPropertiesConfigurationProvider;
import org.mule.runtime.core.api.MuleContext;
import org.mule.runtime.module.observability.AbstractFileObservabilitySignalConfiguration;

import java.io.File;
import java.nio.file.Path;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * A {@link org.mule.runtime.module.observability.configuration.ObservabilitySignalConfiguration} based on a file in the conf
 * folder.
 *
 * @since 4.5.0
 */
public class FileMeterExporterConfiguration extends AbstractFileObservabilitySignalConfiguration {

  private static final String CONFIGURATION_FILE_NAME = "meter-exporter.conf";
  private static final Logger logger = LoggerFactory.getLogger(FileMeterExporterConfiguration.class);
  private final String artifactId;

  private final ConfigurationPropertiesResolver propertyResolver =
      new DefaultConfigurationPropertiesResolver(empty(), new SystemPropertiesConfigurationProvider());

  public FileMeterExporterConfiguration(MuleContext muleContext) {
    // Needs further refactor (the property resolver is instantiated inside the lambda) + We must not receive mule context.
    super((path) -> findArtifactConfigFile(muleContext.getExecutionClassLoader(), path),
          propertyReference -> new DefaultConfigurationPropertiesResolver(empty(),
                                                                          new SystemPropertiesConfigurationProvider())
                                                                              .apply(propertyReference));
    this.artifactId = muleContext.getId();
  }

  @Override
  protected void onConfigurationFileNotFound() {
    logger.atInfo().log("Metric exporter configuration file named '{}' not found {}. Using default configuration.",
                        getSignalConfigurationFileName(), getConfigurationFileLocations());
  }

  private String getConfigurationFileLocations() {
    if (isApplicationLevelConfigurable()) {
      return format("at at both '%s' artifact resources and the '%s' configuration path",
                    artifactId, getSignalConfigurationFileDirectoryPath());
    } else {
      return format("at the '%s' configuration path", getSignalConfigurationFileDirectoryPath());
    }
  }

  @Override
  protected void onConfigurationFileLoadError(Exception error, File configurationFile) {
    logger.atWarn().log("Meter exporter configuration file at '{}' had a parsing error: {}. Using default configuration.",
                        configurationFile.getAbsolutePath(), error.getMessage());
  }

  @Override
  protected String getSignalConfigurationFileName() {
    return CONFIGURATION_FILE_NAME;
  }

  @Override
  protected Path getSignalConfigurationFileDirectoryPath() {
    return get(getProperty(MULE_OPEN_TELEMETRY_METER_EXPORTER_CONFIGURATION_FILE_PATH,
                           getConfFolder().getAbsolutePath())).toAbsolutePath();
  }
}
