/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.metrics.exporter.config.impl;

import static org.mule.runtime.container.api.MuleFoldersUtil.getConfFolder;
import static org.mule.runtime.metrics.exporter.config.api.OpenTelemetryMeterExporterConfigurationProperties.MULE_OPEN_TELEMETRY_METER_EXPORTER_CONFIGURATION_FILE_PATH;

import static java.lang.System.getProperty;
import static java.nio.file.Paths.get;

import org.mule.runtime.core.api.MuleContext;
import org.mule.runtime.module.observability.AbstractFileObservabilitySignalConfiguration;

import java.io.File;
import java.nio.file.Path;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * A {@link org.mule.runtime.module.observability.configuration.ObservabilitySignalConfiguration} based on a file in the conf
 * folder.
 *
 * @since 4.5.0
 */
public class FileMeterExporterConfiguration extends AbstractFileObservabilitySignalConfiguration {

  private static final Logger logger = LoggerFactory.getLogger(FileMeterExporterConfiguration.class);
  private static final String CONFIGURATION_FILE_NAME = "meter-exporter.conf";

  public FileMeterExporterConfiguration(MuleContext muleContext) {
    super(path -> findArtifactConfigFile(muleContext.getExecutionClassLoader(), path),
          muleContext.getDeploymentProperties(), muleContext.getId());
  }

  @Override
  protected void onConfigurationFileNotFound() {
    logger.atInfo().log("Metric exporter configuration file named '{}' not found {}. Using default configuration.",
                        getSignalConfigurationFileName(), getConfigurationFileLocations());
  }

  @Override
  protected void onConfigurationFileLoadError(Exception error, File configurationFile) {
    logger.atWarn().log("Meter exporter configuration file at '{}' had a parsing error: {}. Using default configuration.",
                        configurationFile.getAbsolutePath(), error.getMessage());
  }

  @Override
  protected String getSignalConfigurationFileName() {
    return CONFIGURATION_FILE_NAME;
  }

  @Override
  protected Path getSignalConfigurationFileDirectoryPath() {
    return get(getProperty(MULE_OPEN_TELEMETRY_METER_EXPORTER_CONFIGURATION_FILE_PATH,
                           getConfFolder().getAbsolutePath())).toAbsolutePath();
  }

  @Override
  protected boolean isArtifactLevelProperty(String configurationKey) {
    return false;
  }
}
