/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.metrics.exporter.config.impl;

import static org.mule.runtime.api.util.MuleSystemProperties.ENABLE_OBSERVABILITY_CONFIGURATION_AT_APPLICATION_LEVEL_PROPERTY;
import static org.mule.runtime.metrics.exporter.config.api.OpenTelemetryMeterExporterConfigurationProperties.MULE_OPEN_TELEMETRY_METER_EXPORTER_CA_FILE_LOCATION;
import static org.mule.runtime.metrics.exporter.config.api.OpenTelemetryMeterExporterConfigurationProperties.MULE_OPEN_TELEMETRY_METER_EXPORTER_INTERVAL;
import static org.mule.runtime.metrics.exporter.config.api.OpenTelemetryMeterExporterConfigurationProperties.MULE_OPEN_TELEMETRY_METER_EXPORTER_KEY_FILE_LOCATION;
import static org.mule.test.allure.AllureConstants.Profiling.PROFILING;
import static org.mule.test.allure.AllureConstants.Profiling.ProfilingServiceStory.METRICS_EXPORTER;

import static java.lang.Boolean.TRUE;
import static java.lang.System.setProperty;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.notNullValue;
import static org.hamcrest.Matchers.nullValue;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

import org.mule.runtime.config.internal.model.dsl.config.PropertyNotFoundException;
import org.mule.runtime.core.api.MuleContext;
import org.mule.tck.junit4.rule.SystemProperty;

import java.nio.file.Path;
import java.util.Properties;

import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;

import io.qameta.allure.Feature;
import io.qameta.allure.Story;

@Feature(PROFILING)
@Story(METRICS_EXPORTER)
public class FileMeterExporterConfigurationTestCase {

  public static final String KEY_NON_SYSTEM_PROPERTY = "keyNonSystemProperty";
  public static final String KEY_PROPERTY_SYSTEM_PROPERTY = "keySystemProperty";
  public static final String VALUE_NON_SYSTEM_PROPERTY = "valueNonSystemPropertyConfDirectory";
  public static final String VALUE_SYSTEM_PROPERTY = "valueSystemProperty";
  public static final String SYSTEM_PROPERTY_VALUE = "system_property_value";
  public static final String VALUE_INTERVAL = "20";
  public static final String NO_KEY_IN_FILE = "noKeyInFile";

  private MuleContext muleContext;

  @Rule
  public ExpectedException expectedException = ExpectedException.none();

  @Rule
  public SystemProperty enableConfigurationAtApplicationLevel =
      new SystemProperty(ENABLE_OBSERVABILITY_CONFIGURATION_AT_APPLICATION_LEVEL_PROPERTY, "true");

  @Before
  public void setup() {
    muleContext = mock(MuleContext.class);
    when(muleContext.getDeploymentProperties()).thenReturn(new Properties());
    when(muleContext.getId()).thenReturn("test-artifact");
    when(muleContext.getExecutionClassLoader()).thenReturn(Thread.currentThread().getContextClassLoader());
  }

  @Test
  public void returnsTheResolvedSystemProperty() {
    setProperty(VALUE_SYSTEM_PROPERTY, SYSTEM_PROPERTY_VALUE);
    final FileMeterExporterConfiguration fileSpanExporterConfiguration =
        new TestFileMeterExporterConfiguration(muleContext);
    assertThat(fileSpanExporterConfiguration.getStringValue(KEY_PROPERTY_SYSTEM_PROPERTY), equalTo(
                                                                                                   SYSTEM_PROPERTY_VALUE));
  }

  @Test
  public void whenASystemPropertyCannotBeResolvedAnExceptionIsRaised() {
    expectedException.expect(PropertyNotFoundException.class);
    final FileMeterExporterConfiguration fileSpanExporterConfiguration =
        new TestFileMeterExporterConfiguration(muleContext);
    assertThat(fileSpanExporterConfiguration.getStringValue(KEY_PROPERTY_SYSTEM_PROPERTY), equalTo(
                                                                                                   SYSTEM_PROPERTY_VALUE));
  }

  @Test
  public void returnsTheValueForANonSystemProperty() {
    final FileMeterExporterConfiguration fileMeterExporterConfiguration =
        new TestFileMeterExporterConfiguration(muleContext);
    assertThat(fileMeterExporterConfiguration.getStringValue(KEY_NON_SYSTEM_PROPERTY), equalTo(VALUE_NON_SYSTEM_PROPERTY));
  }

  @Test
  public void whenPropertyIsNotInTheFileNullValueIsReturned() {
    final FileMeterExporterConfiguration fileMeterExporterConfiguration =
        new TestFileMeterExporterConfiguration(muleContext);
    assertThat(fileMeterExporterConfiguration.getStringValue(NO_KEY_IN_FILE), is(nullValue()));
  }

  @Test
  public void whenFileIsNotFoundNoPropertyIsFound() {
    final TestNoFileFoundMeterExporterConfiguration testNoFileFoundSpanExporterConfiguration =
        new TestNoFileFoundMeterExporterConfiguration(muleContext);
    assertThat(testNoFileFoundSpanExporterConfiguration.getStringValue(KEY_NON_SYSTEM_PROPERTY), is(nullValue()));
  }

  @Test
  public void whenValueCorrespondingToPathGetAbsoluteValue() {
    final TestFileMeterExporterConfiguration testfileMeterExporterConfiguration =
        new TestFileMeterExporterConfiguration(muleContext);
    final Path caFileLocationPath =
        testfileMeterExporterConfiguration.getPathValue(MULE_OPEN_TELEMETRY_METER_EXPORTER_CA_FILE_LOCATION);
    final Path keyFileLocationPath =
        testfileMeterExporterConfiguration.getPathValue(MULE_OPEN_TELEMETRY_METER_EXPORTER_KEY_FILE_LOCATION);

    assertThat(caFileLocationPath, is(notNullValue()));
    assertThat(keyFileLocationPath, is(notNullValue()));

    assertThat(caFileLocationPath.isAbsolute(), is(TRUE));
    assertThat(keyFileLocationPath.isAbsolute(), is(TRUE));
  }

  @Test
  public void whenValueCorrespondsToANumberGetsNumberCorrectly() {
    final TestFileMeterExporterConfiguration testfileMeterExporterConfiguration =
        new TestFileMeterExporterConfiguration(muleContext);
    final String interval =
        testfileMeterExporterConfiguration.getStringValue(MULE_OPEN_TELEMETRY_METER_EXPORTER_INTERVAL);

    assertThat(interval, is(equalTo(VALUE_INTERVAL)));
  }

  private static class TestFileMeterExporterConfiguration
      extends FileMeterExporterConfiguration {

    public static final String TEST_CONF_FILE_NAME = "test.conf";

    /**
     * {@link FileMeterExporterConfiguration} used for testing properties file.
     */
    public TestFileMeterExporterConfiguration(MuleContext muleContext) {
      super(muleContext);
    }

    @Override
    protected String getSignalConfigurationFileName() {
      return TEST_CONF_FILE_NAME;
    }
  }

  /**
   * {@link FileMeterExporterConfiguration} used for testing properties file not found.
   */
  private static class TestNoFileFoundMeterExporterConfiguration extends FileMeterExporterConfiguration {

    public static final String TEST_NOT_FOUND_CONF_FILE_NAME = "test-not-found.conf";

    public TestNoFileFoundMeterExporterConfiguration(MuleContext muleContext) {
      super(muleContext);
    }

    @Override
    protected String getSignalConfigurationFileName() {
      return TEST_NOT_FOUND_CONF_FILE_NAME;
    }

  }
}
