/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.metrics.exporter.impl.utils;

import static com.linecorp.armeria.common.HttpResponse.from;
import static com.linecorp.armeria.common.HttpStatus.OK;
import static io.opentelemetry.proto.collector.metrics.v1.ExportMetricsServiceRequest.parseFrom;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.zip.GZIPInputStream;

import com.linecorp.armeria.common.HttpRequest;
import com.linecorp.armeria.common.HttpResponse;
import com.linecorp.armeria.server.AbstractHttpService;
import com.linecorp.armeria.server.ServerBuilder;
import com.linecorp.armeria.server.ServiceRequestContext;
import com.linecorp.armeria.server.grpc.GrpcService;
import com.linecorp.armeria.testing.junit4.server.ServerRule;

import io.grpc.stub.StreamObserver;
import io.opentelemetry.proto.collector.metrics.v1.ExportMetricsServiceRequest;
import io.opentelemetry.proto.collector.metrics.v1.MetricsServiceGrpc;
import org.jetbrains.annotations.NotNull;

import io.opentelemetry.proto.collector.metrics.v1.ExportMetricsServiceResponse;

public class TestServerRule extends ServerRule {

  public static final String GRPC_ENDPOINT_PATH = "/opentelemetry.proto.collector.metrics.v1.MetricsService/Export";
  public static final String HTTP_ENDPOINT_PATH = "/http";

  public static final String HTTP_GZIP_ENDPOINT_PATH = "/http_gzip";

  private List<TestExportedMeter> metrics = new ArrayList<>();
  private final Map<String, String> lastReceivedHeaders = new HashMap<>();

  @Override
  protected void configure(ServerBuilder sb) {
    sb.service(
               GRPC_ENDPOINT_PATH, GrpcService.builder()
                   .addService(
                               new MetricsServiceGrpc.MetricsServiceImplBase() {

                                 @Override
                                 public void export(
                                                    ExportMetricsServiceRequest request,
                                                    StreamObserver<ExportMetricsServiceResponse> responseObserver) {
                                   metrics = TestOpenTelemetryMeterExporterUtils.getMetrics(request);
                                   responseObserver.onNext(ExportMetricsServiceResponse.getDefaultInstance());
                                   responseObserver.onCompleted();
                                 }
                               })
                   .build());


    sb.service(HTTP_ENDPOINT_PATH,
               new AbstractHttpService() {

                 @Override
                 protected @NotNull HttpResponse doPost(@NotNull ServiceRequestContext ctx, @NotNull HttpRequest req) {
                   return HttpResponse.from(req.aggregate().handle((aReq, cause) -> {
                     CompletableFuture<HttpResponse> responseFuture = new CompletableFuture<>();
                     HttpResponse res = from(responseFuture);
                     try {
                       // Capture headers
                       lastReceivedHeaders.clear();
                       aReq.headers().forEach(header -> lastReceivedHeaders.put(header.getKey().toString(),
                                                                                header.getValue().toString()));
                       metrics = TestOpenTelemetryMeterExporterUtils.getMetrics(parseFrom(aReq.content().array()));
                     } catch (IOException e) {
                       // Nothing to do.
                     }
                     responseFuture.complete(HttpResponse.of(OK));
                     return res;
                   }));
                 }
               });

    sb.service(HTTP_GZIP_ENDPOINT_PATH,
               new AbstractHttpService() {

                 @Override
                 protected @NotNull HttpResponse doPost(@NotNull ServiceRequestContext ctx, @NotNull HttpRequest req) {
                   return HttpResponse.from(req.aggregate().handle((aReq, cause) -> {
                     CompletableFuture<HttpResponse> responseFuture = new CompletableFuture<>();
                     HttpResponse res = from(responseFuture);
                     try (GZIPInputStream gzipInputStream = new GZIPInputStream(new ByteArrayInputStream(aReq.content().array()));
                         ByteArrayOutputStream byteArrayOutputStream = new ByteArrayOutputStream()) {

                       byte[] buffer = new byte[1024];
                       int len;
                       while ((len = gzipInputStream.read(buffer)) != -1) {
                         byteArrayOutputStream.write(buffer, 0, len);
                       }
                       byte[] decompressedData = byteArrayOutputStream.toByteArray();
                       metrics = TestOpenTelemetryMeterExporterUtils.getMetrics(parseFrom(decompressedData));
                     } catch (IOException e) {
                       // Nothing to do.
                     }
                     responseFuture.complete(HttpResponse.of(OK));
                     return res;
                   }));
                 }
               });


    sb.http(0);

  }

  public void reset() {
    metrics.clear();
    lastReceivedHeaders.clear();
  }

  public List<TestExportedMeter> getMetrics() {
    return metrics;
  }

  public Map<String, String> getLastReceivedHeaders() {
    return lastReceivedHeaders;
  }

}
