/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.metrics.exporter.impl.capturer;

import static java.util.List.copyOf;

import java.util.Collection;
import java.util.List;
import java.util.Queue;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedQueue;

import io.opentelemetry.sdk.common.CompletableResultCode;
import io.opentelemetry.sdk.metrics.InstrumentType;
import io.opentelemetry.sdk.metrics.data.AggregationTemporality;
import io.opentelemetry.sdk.metrics.data.MetricData;
import io.opentelemetry.sdk.metrics.export.MetricExporter;

/**
 * A {@link MetricExporter} that captures OpenTelemetry exported metrics.
 *
 * @since 4.5.0
 */
public class CapturingMeterExporterWrapper implements MetricExporter {

  private final Set<InMemoryMetricExporter> meterSniffers = ConcurrentHashMap.newKeySet();

  @Override
  public CompletableResultCode export(Collection<MetricData> metrics) {
    meterSniffers.forEach(sniffer -> sniffer.export(metrics));
    return CompletableResultCode.ofSuccess();
  }

  @Override
  public CompletableResultCode flush() {
    return null;
  }

  @Override
  public CompletableResultCode shutdown() {
    return null;
  }

  @Override
  public AggregationTemporality getAggregationTemporality(InstrumentType instrumentType) {
    return AggregationTemporality.CUMULATIVE;
  }

  public InMemoryMetricExporter getExportedMeterSniffer() {
    InMemoryMetricExporter meterSniffer = InMemoryMetricExporter.create();
    meterSniffers.add(meterSniffer);
    return meterSniffer;
  }

  public void dispose(InMemoryMetricExporter meterSniffer) {
    meterSniffers.remove(meterSniffer);
  }

  public static final class InMemoryMetricExporter implements MetricExporter {

    private final Queue<MetricData> finishedMetricItems = new ConcurrentLinkedQueue<>();
    private final AggregationTemporality aggregationTemporality;
    private boolean isStopped = false;

    private InMemoryMetricExporter(AggregationTemporality aggregationTemporality) {
      this.aggregationTemporality = aggregationTemporality;
    }

    public static InMemoryMetricExporter create() {
      return create(AggregationTemporality.CUMULATIVE);
    }

    public static InMemoryMetricExporter create(AggregationTemporality aggregationTemporality) {
      return new InMemoryMetricExporter(aggregationTemporality);
    }

    public List<MetricData> getFinishedMetricItems() {
      return copyOf(this.finishedMetricItems);
    }

    public void reset() {
      this.finishedMetricItems.clear();
    }

    public AggregationTemporality getAggregationTemporality(InstrumentType instrumentType) {
      return this.aggregationTemporality;
    }

    public CompletableResultCode export(Collection<MetricData> metrics) {
      if (this.isStopped) {
        return CompletableResultCode.ofFailure();
      } else {
        this.finishedMetricItems.addAll(metrics);
        return CompletableResultCode.ofSuccess();
      }
    }

    public CompletableResultCode flush() {
      return CompletableResultCode.ofSuccess();
    }

    public CompletableResultCode shutdown() {
      this.isStopped = true;
      this.finishedMetricItems.clear();
      return CompletableResultCode.ofSuccess();
    }
  }

}
