/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.artifact.activation.internal.deployable;

import static org.mule.test.allure.AllureConstants.DeploymentConfiguration.DEPLOYMENT_CONFIGURATION;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.is;

import org.mule.runtime.module.artifact.activation.api.deployable.DeployableProjectModel;
import org.mule.runtime.module.artifact.activation.internal.maven.MavenDeployableProjectModelBuilder;
import org.mule.tck.junit4.AbstractMuleTestCase;

import java.io.File;
import java.net.URISyntaxException;

import io.qameta.allure.Description;
import io.qameta.allure.Feature;
import io.qameta.allure.Issue;
import org.junit.jupiter.api.Test;
import org.junitpioneer.jupiter.SetSystemProperty;

@Feature(DEPLOYMENT_CONFIGURATION)
class DeployableProjectModelVersionParseTestCase extends AbstractMuleTestCase {

  @Test
  @Issue("W-15228973")
  @Description("When a version is provided using pom properties, include the information when building DeployableProjectModel.")
  void buildDeployableProjectModelWithPomProperty() throws Exception {
    DeployableProjectModel deployableProjectModel = getDeployableProjectModel("apps/test-app");
    assertThat(deployableProjectModel.getDescriptor().getVersion(), is("1.2.0-POM"));
  }

  @Test
  @SetSystemProperty(key = "revision", value = "0-SYSTEM")
  @Issue("W-15228973")
  @Description("When a version is provided using system properties, include the information when building DeployableProjectModel.")
  void buildDeployableProjectModelWithSystemProperty() throws Throwable {
    DeployableProjectModel deployableProjectModel = getDeployableProjectModel("apps/test-app");
    assertThat(deployableProjectModel.getDescriptor().getVersion(), is("1.2.0-SYSTEM"));
  }

  @Test
  @Issue("W-19408133")
  @Description("When a version is provided using a property defined within the POM, include the information when building DeployableProjectModel.")
  void buildDeployableProjectModelWithPomPropertyDefinedWithinPom() throws URISyntaxException {
    DeployableProjectModel deployableProjectModel = getDeployableProjectModel("apps/test-app-property-in-pom");
    assertThat(deployableProjectModel.getDescriptor().getVersion(), is("1.2.0-WITHIN-POM"));
  }

  private DeployableProjectModel getDeployableProjectModel(String deployablePath,
                                                           boolean exportAllResourcesAndPackagesIfEmptyLoaderDescriptor,
                                                           boolean includeTestDependencies)
      throws URISyntaxException {
    DeployableProjectModel model =
        new MavenDeployableProjectModelBuilder(getDeployableFolder(deployablePath),
                                               exportAllResourcesAndPackagesIfEmptyLoaderDescriptor, includeTestDependencies)
                                                   .build();

    model.validate();

    return model;
  }

  private DeployableProjectModel getDeployableProjectModel(String deployablePath) throws URISyntaxException {
    return getDeployableProjectModel(deployablePath, false, false);
  }

  protected File getDeployableFolder(String deployableArtifactPath) throws URISyntaxException {
    return new File(getClass().getClassLoader().getResource(deployableArtifactPath).toURI());
  }

}
