/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.artifact.activation.internal.deployable;

import static org.mule.runtime.api.artifact.ArtifactType.PLUGIN;
import static org.mule.runtime.api.util.Preconditions.checkArgument;
import static org.mule.runtime.module.artifact.activation.internal.classloader.model.utils.ArtifactUtils.updatePackagesResources;
import static org.mule.runtime.module.artifact.activation.internal.classloader.model.utils.VersionUtils.getMajor;

import static java.util.stream.Collectors.toList;

import static org.apache.commons.lang3.Strings.CS;

import org.mule.runtime.module.artifact.api.descriptor.BundleDependency;
import org.mule.runtime.module.artifact.api.descriptor.BundleDescriptor;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Optional;

/**
 * Resolves the dependencies of a deployable as {@link BundleDependency artifacts}.
 *
 * @since 4.5
 */
public class DeployablePluginsDependenciesResolver {

  public final Map<BundleDescriptor, List<BundleDependency>> resolve(List<BundleDependency> deployableDependencies) {
    Map<BundleDescriptor, List<BundleDependency>> pluginsDependencies = new HashMap<>();

    List<BundleDependency> dependencies = deployableDependencies.stream()
        .filter(dep -> dep.getDescriptor().getClassifier().isPresent())
        .filter(dep -> PLUGIN.getMavenArtifactClassifier().equals(dep.getDescriptor().getClassifier().get()))
        .collect(toList());

    Map<BundleDependency, List<BundleDependency>> dependenciesMap = resolveDependencies(dependencies);

    for (Map.Entry<BundleDependency, List<BundleDependency>> dependencyListEntry : dependenciesMap.entrySet()) {
      List<BundleDependency> dependencyDependencies =
          resolveConflicts(dependencyListEntry.getValue(), dependencies);
      pluginsDependencies.put(dependencyListEntry.getKey().getDescriptor(),
                              updatePackagesResources(dependencyDependencies));
    }

    return pluginsDependencies;
  }

  private List<BundleDependency> resolveConflicts(List<BundleDependency> newDependencies,
                                                  List<BundleDependency> alreadyResolved) {
    return resolveMulePluginsVersions(newDependencies, alreadyResolved);
  }

  /**
   * Resolve each of the mule plugins dependencies.
   *
   * @param mulePlugins the list of mule plugins that are going to have their dependencies resolved.
   */
  public Map<BundleDependency, List<BundleDependency>> resolveDependencies(List<BundleDependency> mulePlugins) {
    Map<BundleDependency, List<BundleDependency>> muleDependenciesDependencies = new LinkedHashMap<>();
    for (BundleDependency muleDependency : mulePlugins) {
      muleDependenciesDependencies.put(muleDependency, collectTransitiveDependencies(muleDependency));
    }
    return muleDependenciesDependencies;
  }

  private List<BundleDependency> collectTransitiveDependencies(BundleDependency rootDependency) {
    List<BundleDependency> allTransitiveDependencies = new LinkedList<>();
    for (BundleDependency transitiveDependency : rootDependency.getTransitiveDependenciesList()) {
      allTransitiveDependencies.add(transitiveDependency);
      if (transitiveDependency.getDescriptor().getClassifier()
          .map(c -> !PLUGIN.getMavenArtifactClassifier().equals(c))
          .orElse(true)) {
        allTransitiveDependencies.addAll(collectTransitiveDependencies(transitiveDependency));
      }
    }
    return allTransitiveDependencies;
  }

  protected List<BundleDependency> resolveMulePluginsVersions(List<BundleDependency> mulePluginsToResolve,
                                                              List<BundleDependency> definitiveMulePlugins) {
    List<BundleDependency> resolvedPlugins = new ArrayList<>();
    checkArgument(mulePluginsToResolve != null, "List of mule plugins to resolve should not be null");
    checkArgument(definitiveMulePlugins != null, "List of definitive mule plugins should not be null");

    for (BundleDependency mulePluginToResolve : mulePluginsToResolve) {
      Optional<BundleDependency> mulePlugin =
          definitiveMulePlugins.stream().filter(p -> hasSameArtifactIdAndMajor(p, mulePluginToResolve)).findFirst();
      resolvedPlugins.add(mulePlugin.orElse(mulePluginToResolve));
    }
    return resolvedPlugins;
  }

  protected boolean hasSameArtifactIdAndMajor(BundleDependency bundleDependency, BundleDependency otherBundleDependency) {
    BundleDescriptor descriptor = bundleDependency.getDescriptor();
    BundleDescriptor otherDescriptor = otherBundleDependency.getDescriptor();
    return CS.equals(descriptor.getArtifactId(), (CharSequence) otherDescriptor.getArtifactId())
        && CS.equals(getMajor(descriptor.getBaseVersion()), (CharSequence) getMajor(otherDescriptor.getBaseVersion()));
  }

}

