/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.artifact.activation.internal.maven;

import static org.mule.runtime.api.artifact.ArtifactType.APP;
import static org.mule.runtime.api.artifact.ArtifactType.DOMAIN;
import static org.mule.runtime.api.i18n.I18nMessageFactory.createStaticMessage;
import static org.mule.runtime.globalconfig.api.GlobalConfigLoader.getMavenConfig;
import static org.mule.runtime.module.artifact.activation.api.deployable.ArtifactModelResolver.applicationModelResolver;
import static org.mule.runtime.module.artifact.activation.api.deployable.ArtifactModelResolver.domainModelResolver;
import static org.mule.runtime.module.artifact.api.descriptor.ArtifactDescriptorConstants.EXPORTED_PACKAGES;
import static org.mule.runtime.module.artifact.api.descriptor.ArtifactDescriptorConstants.EXPORTED_RESOURCES;
import static org.mule.runtime.module.artifact.api.descriptor.ArtifactDescriptorConstants.INCLUDE_TEST_DEPENDENCIES;

import static java.lang.Boolean.parseBoolean;
import static java.nio.file.Files.find;
import static java.util.Collections.emptyList;
import static java.util.Optional.empty;
import static java.util.stream.Collectors.toList;

import org.mule.maven.pom.parser.api.MavenPomParser;
import org.mule.runtime.api.artifact.ArtifactCoordinates;
import org.mule.runtime.api.artifact.ArtifactType;
import org.mule.runtime.api.deployment.meta.MuleDeployableModel;
import org.mule.runtime.api.exception.MuleRuntimeException;
import org.mule.runtime.module.artifact.activation.api.deployable.DeployableProjectModel;
import org.mule.runtime.module.artifact.api.descriptor.BundleDescriptor;

import java.io.File;
import java.io.IOException;
import java.nio.file.Path;
import java.util.List;
import java.util.Optional;
import java.util.function.Supplier;
import java.util.stream.Stream;

public class LightweightDeployableProjectModelBuilder extends AbstractMavenDeployableProjectModelBuilder {

  private final ArtifactType artifactType;
  private final Optional<MuleDeployableModel> model;

  /**
   * @deprecated Use {@link LightweightDeployableProjectModelBuilder#LightweightDeployableProjectModelBuilder(Path, ArtifactType)}
   *             instead.
   */
  @Deprecated(since = "4.11", forRemoval = true)
  public LightweightDeployableProjectModelBuilder(File projectFolder, boolean isDomain) {
    this(projectFolder, empty(), isDomain);
  }

  public LightweightDeployableProjectModelBuilder(Path projectFolder, ArtifactType artifactType) {
    this(projectFolder, empty(), artifactType);
  }

  /**
   * @deprecated Use {@link LightweightDeployableProjectModelBuilder(Path, Optional, ArtifactType)} instead.
   */
  @Deprecated(since = "4.11", forRemoval = true)
  public LightweightDeployableProjectModelBuilder(File projectFolder, Optional<MuleDeployableModel> model, boolean isDomain) {
    super(getMavenConfig(), projectFolder);
    this.model = model;
    this.artifactType = isDomain ? DOMAIN : APP;
  }

  public LightweightDeployableProjectModelBuilder(Path projectFolder, Optional<MuleDeployableModel> model,
                                                  ArtifactType artifactType) {
    super(getMavenConfig(), projectFolder.toFile());
    this.model = model;
    this.artifactType = artifactType;
  }

  @Override
  protected DeployableProjectModel doBuild(MavenPomParser parser, ArtifactCoordinates deployableArtifactCoordinates) {
    Supplier<MuleDeployableModel> deployableModelResolver = getDeployableModelResolver();
    MuleDeployableModel deployableModel = deployableModelResolver.get();

    List<String> exportedPackages =
        getAttribute(deployableModel.getClassLoaderModelLoaderDescriptor().getAttributes(),
                     EXPORTED_PACKAGES);
    List<String> exportedResources =
        getAttribute(deployableModel.getClassLoaderModelLoaderDescriptor().getAttributes(),
                     EXPORTED_RESOURCES);

    return new DeployableProjectModel(exportedPackages, exportedResources, emptyList(),
                                      buildBundleDescriptor(deployableArtifactCoordinates, artifactType),
                                      deployableModelResolver,
                                      empty(), projectFolder, deployableBundleDependencies,
                                      sharedDeployableBundleDescriptors, additionalPluginDependencies);
  }

  @Override
  protected File getPomFromFolder(File projectFolder) {
    File mavenFolder = new File(projectFolder, "META-INF/maven");
    try (Stream<Path> stream = find(mavenFolder.toPath(), 3, (p, m) -> p.getFileName().toString().equals("pom.xml"))) {
      List<Path> pomLists = stream.collect(toList());
      if (pomLists.size() != 1) {
        throw new MuleRuntimeException(createStaticMessage("Could find the pom in " + mavenFolder.toPath()));
      }
      return pomLists.get(0).toFile();
    } catch (IOException e) {
      throw new MuleRuntimeException(e);
    }
  }

  @Override
  protected boolean isIncludeTestDependencies() {
    return parseBoolean(getSimpleAttribute(getDeployableModelResolver().get().getClassLoaderModelLoaderDescriptor()
        .getAttributes(),
                                           INCLUDE_TEST_DEPENDENCIES, "false"));
  }

  private Supplier<MuleDeployableModel> getDeployableModelResolver() {
    return () -> model.orElseGet(() -> {
      if (artifactType.equals(DOMAIN)) {
        return domainModelResolver().resolve(new File(projectFolder, "META-INF/mule-artifact"));
      }
      if (artifactType.equals(APP)) {
        return applicationModelResolver().resolve(new File(projectFolder, "META-INF/mule-artifact"));
      } else {
        throw new IllegalStateException("No model resolver for artifactType " + artifactType);
      }
    });
  }

  private BundleDescriptor buildBundleDescriptor(ArtifactCoordinates artifactCoordinates, ArtifactType artifactType) {
    return new BundleDescriptor.Builder()
        .setArtifactId(artifactCoordinates.getArtifactId())
        .setGroupId(artifactCoordinates.getGroupId())
        .setVersion(artifactCoordinates.getVersion())
        .setBaseVersion(artifactCoordinates.getVersion())
        .setType(PACKAGE_TYPE)
        .setClassifier(artifactType.getMavenArtifactClassifier())
        .build();
  }

}
