/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.artifact.activation.api.deployable;

import static org.mule.runtime.api.artifact.ArtifactType.PLUGIN;
import static org.mule.runtime.api.util.MuleSystemProperties.API_CLASSIFIERS;
import static org.mule.runtime.module.artifact.api.descriptor.ArtifactConstants.getApiClassifiers;
import static org.mule.test.allure.AllureConstants.ClassloadingIsolationFeature.CLASSLOADING_ISOLATION;
import static org.mule.test.allure.AllureConstants.ClassloadingIsolationFeature.ClassloadingIsolationStory.ARTIFACT_DESCRIPTORS;

import static java.util.Arrays.asList;
import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static java.util.Collections.emptySet;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.containsInAnyOrder;
import static org.hamcrest.core.Is.is;
import static org.hamcrest.core.StringContains.containsString;

import static org.junit.jupiter.api.Assertions.assertThrows;

import org.mule.runtime.module.artifact.activation.api.ArtifactActivationException;
import org.mule.runtime.module.artifact.api.descriptor.BundleDependency;
import org.mule.runtime.module.artifact.api.descriptor.BundleDescriptor;
import org.mule.tck.junit4.AbstractMuleTestCase;

import java.io.File;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junitpioneer.jupiter.SetSystemProperty;

import io.qameta.allure.Feature;
import io.qameta.allure.Issue;
import io.qameta.allure.Story;

@Feature(CLASSLOADING_ISOLATION)
@Story(ARTIFACT_DESCRIPTORS)
class DeployableProjectModelValidationsTestCase extends AbstractMuleTestCase {

  private BundleDescriptor appDescriptor;

  @BeforeEach
  public void setUp() {
    appDescriptor = new BundleDescriptor.Builder()
        .setGroupId("org.mule.sample")
        .setArtifactId("test-app")
        .setVersion("0.0.1")
        .setClassifier("mule-application")
        .build();
  }

  @Test
  void sharedLibraryNotInProject() {
    List<BundleDependency> dependencies = new ArrayList<>();

    dependencies.add(new BundleDependency.Builder()
        .setDescriptor(new BundleDescriptor.Builder()
            .setGroupId("org.mule.sample")
            .setArtifactId("test-dep-a")
            .setVersion("0.0.1")
            .build())
        .build());

    Set<BundleDescriptor> sharedLibraries = new HashSet<>();
    sharedLibraries.add(new BundleDescriptor.Builder()
        .setGroupId("org.mule.sample")
        .setArtifactId("test-dep-b")
        .setVersion("0.0.1")
        .build());

    var model = new DeployableProjectModel(emptyList(), emptyList(), emptyList(),
                                           appDescriptor,
                                           () -> null,
                                           new File("."),
                                           dependencies,
                                           sharedLibraries,
                                           emptyMap());
    var thrown = assertThrows(ArtifactActivationException.class, () -> model.validate());
    assertThat(thrown.getMessage(),
               is(containsString(" * Artifact 'org.mule.sample:test-dep-b' is declared as a sharedLibrary but is not a dependency of the project")));
  }

  @Test
  void sharedLibraryInProject() {
    BundleDescriptor testDep = new BundleDescriptor.Builder()
        .setGroupId("org.mule.sample")
        .setArtifactId("test-dep-a")
        .setVersion("0.0.1")
        .build();

    List<BundleDependency> dependencies = new ArrayList<>();
    dependencies.add(new BundleDependency.Builder()
        .setDescriptor(testDep)
        .build());

    Set<BundleDescriptor> sharedLibraries = new HashSet<>();
    sharedLibraries.add(testDep);

    new DeployableProjectModel(emptyList(), emptyList(), emptyList(),
                               appDescriptor,
                               () -> null,
                               new File("."),
                               dependencies,
                               sharedLibraries,
                               emptyMap()).validate();
  }

  @Test
  void pluginForAdditionalDependenciesNotInProject() {
    List<BundleDependency> dependencies = new ArrayList<>();
    dependencies.add(new BundleDependency.Builder()
        .setDescriptor(new BundleDescriptor.Builder()
            .setGroupId("org.mule.sample")
            .setArtifactId("test-plugin-z")
            .setVersion("0.0.1")
            .setClassifier(PLUGIN.getMavenArtifactClassifier())
            .build())
        .build());

    Map<BundleDescriptor, List<BundleDependency>> additionalPluginDependencies = new HashMap<>();
    additionalPluginDependencies.put(new BundleDescriptor.Builder()
        .setGroupId("org.mule.sample")
        .setArtifactId("test-plugin-a")
        .setVersion("0.0.1")
        .setClassifier(PLUGIN.getMavenArtifactClassifier())
        .build(), asList(new BundleDependency.Builder()
            .setDescriptor(new BundleDescriptor.Builder()
                .setGroupId("org.mule.sample")
                .setArtifactId("test-dep-a")
                .setVersion("0.0.1")
                .build())
            .build()));

    var model = new DeployableProjectModel(emptyList(), emptyList(), emptyList(),
                                           appDescriptor,
                                           () -> null,
                                           new File("."),
                                           dependencies,
                                           emptySet(),
                                           additionalPluginDependencies);
    var thrown = assertThrows(ArtifactActivationException.class, () -> model.validate());
    assertThat(thrown.getMessage(),
               is(containsString(" * Mule Plugin 'org.mule.sample:test-plugin-a' is declared in additionalPluginDependencies but is not a dependency of the project")));
  }

  @Test
  void pluginForAdditionalDependenciesInProject() {
    BundleDescriptor testPlugin = new BundleDescriptor.Builder()
        .setGroupId("org.mule.sample")
        .setArtifactId("test-plugin-a")
        .setVersion("0.0.1")
        .setClassifier(PLUGIN.getMavenArtifactClassifier())
        .build();

    List<BundleDependency> dependencies = new ArrayList<>();
    dependencies.add(new BundleDependency.Builder()
        .setDescriptor(testPlugin)
        .build());

    Map<BundleDescriptor, List<BundleDependency>> additionalPluginDependencies = new HashMap<>();
    additionalPluginDependencies.put(testPlugin, asList(new BundleDependency.Builder()
        .setDescriptor(new BundleDescriptor.Builder()
            .setGroupId("org.mule.sample")
            .setArtifactId("test-dep-a")
            .setVersion("0.0.1")
            .build())
        .build()));

    new DeployableProjectModel(emptyList(), emptyList(), emptyList(),
                               appDescriptor,
                               () -> null,
                               new File("."),
                               dependencies,
                               emptySet(),
                               additionalPluginDependencies).validate();
  }

  @Test
  @Issue("W-11202204")
  void conflictingPluginVersions() {
    List<BundleDependency> dependencies = new ArrayList<>();

    dependencies.add(new BundleDependency.Builder()
        .setDescriptor(new BundleDescriptor.Builder()
            .setGroupId("org.mule.sample")
            .setArtifactId("test-plugin-a")
            .setVersion("0.0.1")
            .setClassifier(PLUGIN.getMavenArtifactClassifier())
            .build())
        .build());
    dependencies.add(new BundleDependency.Builder()
        .setDescriptor(new BundleDescriptor.Builder()
            .setGroupId("org.mule.sample")
            .setArtifactId("test-plugin-a")
            .setVersion("0.1.0")
            .setClassifier(PLUGIN.getMavenArtifactClassifier())
            .build())
        .build());

    var model = new DeployableProjectModel(emptyList(), emptyList(), emptyList(),
                                           appDescriptor,
                                           () -> null,
                                           new File("."),
                                           dependencies,
                                           emptySet(),
                                           emptyMap());
    var thrown = assertThrows(ArtifactActivationException.class, () -> model.validate());
    assertThat(thrown.getMessage(),
               is(containsString(" * Mule Plugin 'org.mule.sample:test-plugin-a:mule-plugin' is depended upon in the project multiple times with versions ('0.0.1, 0.1.0') in the dependency graph.")));
  }

  @Test
  void nonConflictingPluginVersions() {
    List<BundleDependency> dependencies = new ArrayList<>();

    dependencies.add(new BundleDependency.Builder()
        .setDescriptor(new BundleDescriptor.Builder()
            .setGroupId("org.mule.sample")
            .setArtifactId("test-plugin-1")
            .setVersion("0.0.1")
            .setClassifier(PLUGIN.getMavenArtifactClassifier())
            .build())
        .build());
    dependencies.add(new BundleDependency.Builder()
        .setDescriptor(new BundleDescriptor.Builder()
            .setGroupId("org.mule.sample")
            .setArtifactId("test-plugin-a")
            .setVersion("0.1.1")
            .setClassifier(PLUGIN.getMavenArtifactClassifier())
            .build())
        .build());

    var model = new DeployableProjectModel(emptyList(), emptyList(), emptyList(),
                                           appDescriptor,
                                           () -> null,
                                           new File("."),
                                           dependencies,
                                           emptySet(),
                                           emptyMap());
    model.validate();
  }

  @Test
  @Issue("W-12118812")
  void nonConflictingPluginsWithSameGroupIdAndArtifactIdButDifferentClassifier() {
    List<BundleDependency> dependencies = new ArrayList<>();

    dependencies.add(new BundleDependency.Builder()
        .setDescriptor(new BundleDescriptor.Builder()
            .setGroupId("org.mule.sample")
            .setArtifactId("test-plugin-a")
            .setVersion("0.0.1")
            .setClassifier(PLUGIN.getMavenArtifactClassifier())
            .build())
        .build());
    dependencies.add(new BundleDependency.Builder()
        .setDescriptor(new BundleDescriptor.Builder()
            .setGroupId("org.mule.sample")
            .setArtifactId("test-plugin-a")
            .setVersion("0.1.0")
            .setClassifier(null)
            .build())
        .build());

    var model = new DeployableProjectModel(emptyList(), emptyList(), emptyList(),
                                           appDescriptor,
                                           () -> null,
                                           new File("."),
                                           dependencies,
                                           emptySet(),
                                           emptyMap());
    model.validate();
  }

  @Test
  @Issue("W-12395077")
  void noFailureWithApiDependencies() {
    for (String classifier : getApiClassifiers()) {
      List<BundleDependency> dependencies = new ArrayList<>();

      dependencies.add(new BundleDependency.Builder()
          .setDescriptor(new BundleDescriptor.Builder()
              .setGroupId("org.mule.sample")
              .setArtifactId("test-api-a")
              .setVersion("0.0.1")
              .setClassifier(classifier)
              .build())
          .build());
      dependencies.add(new BundleDependency.Builder()
          .setDescriptor(new BundleDescriptor.Builder()
              .setGroupId("org.mule.sample")
              .setArtifactId("test-api-a")
              .setVersion("0.1.0")
              .setClassifier(classifier)
              .build())
          .build());

      var model = new DeployableProjectModel(emptyList(), emptyList(), emptyList(),
                                             appDescriptor,
                                             () -> null,
                                             new File("."),
                                             dependencies,
                                             emptySet(),
                                             emptyMap());
      model.validate();
    }
  }

  @Test
  @Issue("W-12395077")
  @SetSystemProperty(key = API_CLASSIFIERS, value = " custom-classifier-a , custom-classifier-b  ")
  void noFailureWithCustomApiDependencies() {
    Set<String> apiClassifiers = getApiClassifiers();

    assertThat(apiClassifiers, containsInAnyOrder("custom-classifier-a", "custom-classifier-b"));

    for (String classifier : apiClassifiers) {
      List<BundleDependency> dependencies = new ArrayList<>();

      dependencies.add(new BundleDependency.Builder()
          .setDescriptor(new BundleDescriptor.Builder()
              .setGroupId("org.mule.sample")
              .setArtifactId("test-api-a")
              .setVersion("0.0.1")
              .setClassifier(classifier)
              .build())
          .build());
      dependencies.add(new BundleDependency.Builder()
          .setDescriptor(new BundleDescriptor.Builder()
              .setGroupId("org.mule.sample")
              .setArtifactId("test-api-a")
              .setVersion("0.1.0")
              .setClassifier(classifier)
              .build())
          .build());

      var model = new DeployableProjectModel(emptyList(), emptyList(), emptyList(),
                                             appDescriptor,
                                             () -> null,
                                             new File("."),
                                             dependencies,
                                             emptySet(),
                                             emptyMap());
      model.validate();
    }
  }
}
