/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.artifact.activation.internal.descriptor;

import static org.mule.runtime.api.util.xmlsecurity.XMLSecureFactories.createDefault;

import static java.lang.String.format;

import org.mule.runtime.module.artifact.activation.api.descriptor.MuleConfigurationsFilter;

import java.io.File;
import java.io.FileInputStream;
import java.io.InputStream;
import java.nio.file.Path;

import javax.xml.namespace.QName;
import javax.xml.stream.XMLEventReader;
import javax.xml.stream.XMLInputFactory;
import javax.xml.stream.events.StartElement;
import javax.xml.stream.events.XMLEvent;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class XmlMuleConfigurationsFilter implements MuleConfigurationsFilter {

  private static final Logger LOGGER = LoggerFactory.getLogger(XmlMuleConfigurationsFilter.class);

  private static final String DEFAULT_NAMESPACE_URI_MASK = "http://www.mulesoft.org/schema/mule/%s";
  private static final String CORE_NAMESPACE = format(DEFAULT_NAMESPACE_URI_MASK, "core");
  private static final String DOMAIN_NAMESPACE = format(DEFAULT_NAMESPACE_URI_MASK, "domain");
  private static final String CONFIG_FILE_EXTENSION = ".xml";
  private static final String muleLocalName = "mule";
  private static final String muleDomainLocalName = "mule-domain";

  private final XMLInputFactory factory = createDefault().getXMLInputFactory();

  @Override
  public boolean filter(File candidateConfig) {
    if (!candidateConfig.getName().endsWith(CONFIG_FILE_EXTENSION)) {
      return false;
    }

    QName documentName;
    try {
      documentName = generateDocument(candidateConfig.toPath());
    } catch (Exception e) {
      LOGGER.error("Exception parsing app xml config", e);
      return false;
    }

    if (documentName == null) {
      return false;
    }

    return hasMuleAsRootElement(documentName);
  }

  private boolean hasMuleAsRootElement(QName docRootTagName) {
    String rootElementLocalName = docRootTagName.getLocalPart();
    String rootElementNamespace = docRootTagName.getNamespaceURI();

    return (rootElementLocalName.equals(muleLocalName) && rootElementNamespace.equals(CORE_NAMESPACE))
        || (rootElementLocalName.equals(muleDomainLocalName) && rootElementNamespace.equals(DOMAIN_NAMESPACE));
  }

  private QName generateDocument(Path filePath) throws Exception {
    XMLEventReader eventReader = null;
    try (InputStream fis = new FileInputStream(filePath.toFile())) {
      eventReader = factory.createXMLEventReader(fis);

      while (eventReader.hasNext()) {
        XMLEvent event = eventReader.nextEvent();

        if (event.isStartElement()) {
          // early return
          return ((StartElement) event).getName();
        }
      }
      return null;
    } finally {
      if (eventReader == null) {
        return null;
      }
      eventReader.close();
    }
  }
}
