/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.artifact.internal.util;

import static java.nio.charset.StandardCharsets.UTF_8;
import static java.nio.file.Files.createTempFile;

import static org.apache.commons.io.FileUtils.writeStringToFile;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.core.IsIterableContaining.hasItem;

import org.mule.tck.ZipUtils;
import org.mule.tck.ZipUtils.ZipResource;
import org.mule.tck.junit4.AbstractMuleTestCase;

import java.io.File;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Set;

import org.junit.jupiter.api.Test;

public class FileJarExplorerTestCase extends AbstractMuleTestCase {

  private final FileJarExplorer packageExplorer = new FileJarExplorer();

  @Test
  void readsPackagesFromJar() throws Exception {
    final ZipResource fooClass = new ZipResource("EchoTest.clazz", "org/foo/Foo.class");
    final ZipResource barClass = new ZipResource("EchoTest.clazz", "org/bar/Bar.class");
    final ZipResource[] zipResources = {fooClass, barClass};

    final Path jarFile = createTempFile("test", ".jar");
    Files.delete(jarFile);
    ZipUtils.compress(jarFile, zipResources);

    final Set<String> packages = packageExplorer.explore(jarFile.toUri()).getPackages();
    assertThat(packages.size(), equalTo(2));
    assertThat(packages, hasItem("org.foo"));
    assertThat(packages, hasItem("org.bar"));
  }

  @Test
  void readsPackagesFromFolder() throws Exception {
    final File folder = File.createTempFile("test", "");
    folder.delete();
    folder.mkdirs();
    final File orgFolder = new File(folder, "org");
    final File orgFooFolder = new File(orgFolder, "foo");
    final File orgFooBarFolder = new File(orgFolder, "bar");
    writeStringToFile(new File(orgFooFolder, "Foo.class"), "foo", UTF_8);
    writeStringToFile(new File(orgFooBarFolder, "Bar.class"), "bar", UTF_8);

    final Set<String> packages = packageExplorer.explore(folder.toURI()).getPackages();
    assertThat(packages.size(), equalTo(2));
    assertThat(packages, hasItem("org.foo"));
    assertThat(packages, hasItem("org.bar"));
  }

  @Test
  void readsResourcesFromFolder() throws Exception {
    final File folder = File.createTempFile("test", "");
    folder.delete();
    folder.mkdirs();
    final File orgFolder = new File(folder, "org");
    final File orgFooFolder = new File(orgFolder, "foo");
    final File orgFooBarFolder = new File(orgFolder, "bar");
    writeStringToFile(new File(orgFooFolder, "foo.txt"), "foo", UTF_8);
    writeStringToFile(new File(orgFooBarFolder, "bar.txt"), "bar", UTF_8);

    final Set<String> resources = packageExplorer.explore(folder.toURI()).getResources();
    assertThat(resources.size(), equalTo(2));
    assertThat(resources, hasItem("org/foo/foo.txt"));
    assertThat(resources, hasItem("org/bar/bar.txt"));
  }
}
