/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.container.internal.splash;

import static org.mule.runtime.api.util.MuleSystemProperties.SYSTEM_PROPERTY_PREFIX;
import static org.mule.runtime.core.api.config.i18n.CoreMessages.failedToConvertStringUsingEncoding;

import static java.lang.Boolean.TRUE;
import static java.lang.String.format;
import static java.lang.System.getProperty;
import static java.lang.System.lineSeparator;
import static java.util.Arrays.asList;

import static org.apache.commons.lang3.StringUtils.repeat;

import org.mule.runtime.api.exception.MuleRuntimeException;
import org.mule.runtime.core.api.MuleContext;

import java.io.UnsupportedEncodingException;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Implements singleton pattern to allow different splash-screen implementations following the concept of header, body, and
 * footer. Header and footer are reserved internally to Mule but body can be used to customize splash-screen output. External code
 * can e.g. hook into the start-up splash-screen as follows:
 *
 * <pre>
 * <code>
 *   SplashScreen splashScreen = SplashScreen.getInstance(ServerStartupSplashScreen.class);
 *   splashScreen.addBody("Some extra text");
 * </code>
 * </pre>
 */
public abstract class SplashScreen {

  public static Logger LOGGER = LoggerFactory.getLogger(SplashScreen.class);

  public static final int DEFAULT_MESSAGE_WIDTH = 80;

  public static final String RUNTIME_VERBOSE = SYSTEM_PROPERTY_PREFIX + "runtime.verbose";
  public static final String CUSTOM_NAMES = SYSTEM_PROPERTY_PREFIX + "splash.masked.properties";

  private static final List<String> CREDENTIAL_NAMES = asList("key", "password", "pswd");
  private static final Set<String> CUSTOM_CREDENTIAL_NAMES = new HashSet<>(asList(getProperty(CUSTOM_NAMES, "").split(",")));
  public static final String CREDENTIAL_MASK = "*****";
  /**
   * Determines whether extra information should be display.
   */
  protected static PropertyChecker RUNTIME_VERBOSE_PROPERTY = new PropertyChecker(RUNTIME_VERBOSE, TRUE.toString());

  protected static final String VALUE_FORMAT = " - %s";
  private static final String KEY_VALUE_FORMAT = VALUE_FORMAT + " = %s";
  protected List<String> header = new ArrayList<>();
  protected List<String> body = new ArrayList<>();
  protected List<String> footer = new ArrayList<>();

  /**
   * Setting the header clears body and footer assuming a new splash-screen is built.
   *
   */
  final public void setHeader(MuleContext context) {
    header.clear();
    doHeader(context);
  }

  final public void addBody(String line) {
    doBody(line);
  }

  final public void setFooter(MuleContext context) {
    footer.clear();
    doFooter(context);
  }

  public static String miniSplash(final String message) {
    // middle dot char
    return " + " + message;
  }

  protected void doHeader(MuleContext context) {
    // default reserved for mule core info
  }

  protected void doBody(String line) {
    body.add(line);
  }

  protected void doFooter(MuleContext context) {
    // default reserved for mule core info
  }

  protected void listItems(Collection<String> items, String description) {
    if (!items.isEmpty()) {
      doBody(description);
      for (String item : items) {
        doBody(format(VALUE_FORMAT, item));
      }
    }
  }

  private boolean isCredentialItem(String key) {
    if (CUSTOM_CREDENTIAL_NAMES.contains(key)) {
      return true;
    }
    for (String credentialName : CREDENTIAL_NAMES) {
      if (key.toLowerCase().contains(credentialName)) {
        return true;
      }
    }
    return false;
  }

  protected void listItems(Map<String, String> map, String description) {
    if (!map.isEmpty()) {
      doBody(description);
      for (String key : map.keySet()) {
        String value = isCredentialItem(key) ? CREDENTIAL_MASK : map.get(key);
        doBody(format(KEY_VALUE_FORMAT, key, value));
      }
    }
  }

  @Override
  public String toString() {
    List<String> messages = new ArrayList<>(header);
    messages.addAll(body);
    messages.addAll(footer);
    return wrapInTextBox(messages, '*', 70);
  }

  private static String wrapInTextBox(List<String> messages, char c, int maxlength) {
    int size;
    StringBuilder buf = new StringBuilder(messages.size() * maxlength);
    int trimLength = maxlength - (c == ' ' ? 2 : 4);

    messages = messages.stream()
        .map(string -> string.split(lineSeparator()))
        .flatMap(Arrays::stream)
        .collect(Collectors.toList());

    for (int i = 0; i < messages.size(); i++) {
      size = messages.get(i).toString().length();
      if (size > trimLength) {
        String temp = messages.get(i).toString();
        int k = i;
        int x;
        int len;
        messages.remove(i);
        while (temp.length() > 0) {
          len = (trimLength <= temp.length() ? trimLength : temp.length());
          String msg = temp.substring(0, len);
          x = msg.indexOf(lineSeparator());

          if (x > -1) {
            msg = msg.substring(0, x);
            len = x + 1;
          } else {
            x = msg.lastIndexOf(' ');
            if (x > -1 && len == trimLength) {
              msg = msg.substring(0, x);
              len = x + 1;
            }
          }
          if (msg.startsWith(" ")) {
            msg = msg.substring(1);
          }

          temp = temp.substring(len);
          messages.add(k, msg);
          k++;
        }
      }
    }

    buf.append(lineSeparator());
    if (c != ' ') {
      buf.append(repeat(c, maxlength));
    }

    for (String message : messages) {
      buf.append(lineSeparator());
      if (c != ' ') {
        buf.append(c);
      }
      buf.append(" ");
      buf.append(message);

      String osEncoding = Charset.defaultCharset().name();
      int padding;
      try {
        padding = trimLength - message.toString().getBytes(osEncoding).length;
      } catch (UnsupportedEncodingException ueex) {
        throw new MuleRuntimeException(failedToConvertStringUsingEncoding(osEncoding), ueex);
      }
      if (padding > 0) {
        buf.append(repeat(" ", padding));
      }
      buf.append(' ');
      if (c != ' ') {
        buf.append(c);
      }
    }
    buf.append(lineSeparator());
    if (c != ' ') {
      buf.append(repeat(c, maxlength));
    }
    return buf.toString();
  }

  protected SplashScreen() {
    // make sure no one else creates an instance
  }
}
