/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.container.api;

import static org.mule.runtime.api.util.Preconditions.checkArgument;
import static org.mule.runtime.core.api.config.MuleProperties.MULE_BASE_DIRECTORY_PROPERTY;
import static org.mule.runtime.core.api.config.MuleProperties.MULE_HOME_DIRECTORY_PROPERTY;

import static java.io.File.separator;
import static java.util.Optional.of;
import static java.util.Optional.ofNullable;

import static org.apache.commons.lang3.StringUtils.isEmpty;

import org.mule.runtime.api.exception.MuleRuntimeException;

import java.io.File;
import java.io.IOException;
import java.nio.file.Path;
import java.util.Optional;

/**
 * Calculates folders for a mule server based on the
 * {@value org.mule.runtime.core.api.config.MuleProperties#MULE_HOME_DIRECTORY_PROPERTY} property
 */
public class MuleFoldersUtil {

  public static final String EXECUTION_FOLDER = ".mule";
  public static final String LIB_FOLDER = "lib";
  public static final String SHARED_FOLDER = "shared";
  public static final String APPS_FOLDER = "apps";
  public static final String PLUGINS_FOLDER = "plugins";
  public static final String DOMAINS_FOLDER = "domains";
  public static final String CONTAINER_APP_PLUGINS = "plugins";
  public static final String SERVER_PLUGINS = "server-plugins";
  public static final String CONF = "conf";
  public static final String USER_FOLDER = "user";
  public static final String PATCHES_FOLDER = "patches";
  public static final String ARTIFACT_PATCHES_FOLDER = "mule-artifact-patches";
  public static final String SERVICES_FOLDER = "services";
  private static final String MODULES_FOLDER = "modules";
  private static final String NATIVE_LIBRARIES_FOLDER = "native-libraries";


  private MuleFoldersUtil() {}

  /**
   * @return the mule runtime installation folder.
   * @deprecated Use {@link #getMuleBaseFolderPath()} instead.
   */
  @Deprecated(forRemoval = false, since = "4.11")
  public static File getMuleHomeFolder() {
    return getMuleHomeFolderPath().toFile();
  }

  /**
   * @return the mule runtime installation folder.
   */
  public static Path getMuleHomeFolderPath() {
    Path muleHome = getMuleHomePath().orElse(null);
    if (muleHome == null) {
      muleHome = Path.of(".");
    }
    return muleHome;
  }

  /**
   * @return the MULE_HOME directory of this instance. Returns null if the property is not set
   */
  private static Optional<Path> getMuleHomePath() {
    final String muleHome = System.getProperty(MULE_HOME_DIRECTORY_PROPERTY);
    return ofNullable(muleHome != null ? Path.of(muleHome) : null);
  }

  /**
   * @return the mule runtime base folder.
   */
  public static File getMuleBaseFolder() {
    return getMuleBaseFolderPath().toFile();
  }

  /**
   * @return the mule runtime base folder.
   */
  public static Path getMuleBaseFolderPath() {
    Path muleBase = getMuleBasePath().orElse(null);
    if (muleBase == null) {
      muleBase = getMuleHomeFolderPath();
    }
    return muleBase;
  }

  /**
   * The mule runtime base folder is a directory similar to the mule runtime installation one but with only the specific
   * configuration parts of the mule runtime installation such as the apps folder, the domain folder, the conf folder.
   *
   * @return the MULE_BASE directory of this instance. Returns the
   *         {@link org.mule.runtime.core.api.config.MuleProperties#MULE_HOME_DIRECTORY_PROPERTY} property value if
   *         {@link org.mule.runtime.core.api.config.MuleProperties#MULE_BASE_DIRECTORY_PROPERTY} is not set which may be null.
   */
  private static Optional<Path> getMuleBasePath() {
    String muleBaseVar = System.getProperty(MULE_BASE_DIRECTORY_PROPERTY);

    if (muleBaseVar != null && !muleBaseVar.trim().equals("") && !muleBaseVar.equals("%MULE_BASE%")) {
      try {
        return of(Path.of(muleBaseVar).toRealPath());
      } catch (IOException e) {
        throw new MuleRuntimeException(e);
      }
    }

    return getMuleHomePath();
  }

  /**
   * @return a {@link File} pointing to the container folder that contains services.
   */
  public static File getServicesFolder() {
    return getServicesFolderPath().toFile();
  }

  /**
   * @return a {@link File} pointing to the container folder that contains services.
   */
  public static Path getServicesFolderPath() {
    return getMuleBaseFolderPath().resolve(SERVICES_FOLDER);
  }

  /**
   * Returns the file for a given service name.
   *
   * @param name name of the service. Non empty.
   * @return a {@link File} pointing to the folder that corresponds to the provided service name when installed.
   */
  public static File getServiceFolder(String name) {
    checkArgument(!isEmpty(name), "name cannot be empty");
    return getServicesFolderPath().resolve(name).toFile();
  }

  /**
   * @return a {@link File} pointing to the folder where server plugins are located.
   */
  public static File getServerPluginsFolder() {
    return new File(getMuleBaseFolder(), SERVER_PLUGINS);
  }

  /**
   * @return a {@link File} pointing to the folder where server plugins are located.
   */
  public static Path getServerPluginsFolderPath() {
    return getMuleBaseFolderPath().resolve(SERVER_PLUGINS);
  }

  /**
   * @return a {@link File} pointing to the folder where the server configuration is located.
   */
  public static File getConfFolder() {
    return new File(getMuleBaseFolder(), CONF);
  }

  public static File getDomainsFolder() {
    return new File(getMuleBaseFolder(), DOMAINS_FOLDER);
  }

  public static Path getDomainsFolderPath() {
    return getMuleBaseFolderPath().resolve(DOMAINS_FOLDER);
  }

  public static File getDomainFolder(String domainName) {
    return new File(getDomainsFolder(), domainName);
  }

  public static File getDomainLibFolder(String domainName) {
    return new File(getDomainFolder(domainName), LIB_FOLDER);
  }

  public static File getAppsFolder() {
    return new File(getMuleBaseFolder(), APPS_FOLDER);
  }

  public static Path getAppsFolderPath() {
    return getMuleBaseFolderPath().resolve(APPS_FOLDER);
  }

  public static File getAppFolder(String appName) {
    return new File(getAppsFolder(), appName);
  }

  public static File getAppDataFolder(String appDataFolder) {
    return new File(getExecutionFolder(), appDataFolder);
  }

  /**
   * @param appName name of the application to look for
   * @return the libraries folder in the deployed application with the given name
   */
  public static File getAppLibFolder(String appName) {
    return new File(getAppFolder(appName), getAppLibsFolderPath());
  }

  /**
   * @param appName name of the application to look for
   * @return the plugins folder in the deployed application with the given name
   */
  public static File getAppPluginsFolder(String appName) {
    return new File(getAppFolder(appName), getAppPluginsFolderPath());
  }

  /**
   * @return relative path for plugins on an application
   */
  public static String getAppPluginsFolderPath() {
    return PLUGINS_FOLDER + separator;
  }

  /**
   * @param appName name of the application to look for
   * @return the shared libraries folder in the deployed application with the given name
   */
  public static File getAppSharedLibsFolder(String appName) {
    return new File(getAppFolder(appName), getAppSharedLibsFolderPath());
  }

  /**
   * @return relative path for shared libraries on an application
   */
  public static String getAppSharedLibsFolderPath() {
    return getAppLibsFolderPath() + SHARED_FOLDER + separator;
  }

  private static String getAppLibsFolderPath() {
    return LIB_FOLDER + separator;
  }

  /**
   * @return relative path for libraries on an application
   */
  public static File getExecutionFolder() {
    return new File(getMuleBaseFolder(), EXECUTION_FOLDER);
  }

  public static Path getExecutionFolderPath() {
    return getMuleBaseFolderPath().resolve(EXECUTION_FOLDER);
  }

  public static File getMuleLibFolder() {
    return new File(getMuleHomeFolder(), LIB_FOLDER);
  }

  public static Path getMuleLibFolderPath() {
    return getMuleHomeFolderPath().resolve(LIB_FOLDER);
  }

  public static File getUserLibFolder() {
    return new File(getMuleLibFolder(), USER_FOLDER);
  }

  /**
   * @return directory where the patches are placed in the runtime
   */
  public static File getPatchesLibFolder() {
    return new File(getMuleLibFolder(), PATCHES_FOLDER);
  }

  /**
   * @return directory where the artifact patches are placed in the runtime
   */
  public static File getArtifactPatchesLibFolder() {
    return new File(getPatchesLibFolder(), ARTIFACT_PATCHES_FOLDER);
  }

  public static File getContainerAppPluginsFolder() {
    return new File(getMuleBaseFolder(), CONTAINER_APP_PLUGINS);
  }

  /**
   * @return a {@link File} pointing to the container folder used to temporarily store services on deployment
   */
  public static File getServicesTempFolder() {
    return new File(getExecutionFolder(), SERVICES_FOLDER);
  }

  /**
   * @return a {@link File} pointing to the container folder used to temporarily store the exported module services for SPI.
   */
  public static File getModulesTempFolder() {
    return new File(getExecutionFolder(), MODULES_FOLDER);
  }

  /**
   * @return a {@link File} pointing to the container folder used to temporarily store the native libraries loaded for the app.
   */
  public static File getNativeLibrariesTempFolder() {
    return new File(getExecutionFolder(), NATIVE_LIBRARIES_FOLDER);
  }

  /**
   * @param appName name of the application to look for
   * @return the native libraries folder in the execution folder with the given name.
   */
  public static File getAppNativeLibrariesTempFolder(String appName) {
    return new File(getNativeLibrariesTempFolder(), appName);
  }

  /**
   * @param appName    name of the application to look for
   * @param identifier of the application instance
   * @return the native libraries folder in the execution folder with the given name.
   */
  public static File getAppNativeLibrariesTempFolder(String appName, String identifier) {
    return new File(new File(getNativeLibrariesTempFolder(), appName), identifier);
  }
}
